"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = {
    openapi: "3.0.3",
    info: {
        title: "Identity Toolkit API",
        description: "The Google Identity Toolkit API lets you use open standards to verify a user's identity.",
        contact: { name: "Google", url: "https://google.com" },
        version: "v1",
        license: {
            name: "Creative Commons Attribution 3.0",
            url: "http://creativecommons.org/licenses/by/3.0/",
        },
        termsOfService: "https://developers.google.com/terms/",
    },
    servers: [{ url: "https://identitytoolkit.googleapis.com" }],
    externalDocs: { url: "https://cloud.google.com/identity-platform" },
    tags: [
        { name: "accounts" },
        { name: "projects" },
        { name: "v1" },
        { name: "defaultSupportedIdps" },
        { name: "secureToken" },
        { name: "emulator" },
    ],
    paths: {
        "/v1/accounts:createAuthUri": {
            post: {
                description: "If an email identifier is specified, checks and returns if any user account is registered with the email. If there is a registered account, fetches all providers associated with the account's email. If the provider ID of an Identity Provider (IdP) is specified, creates an authorization URI for the IdP. The user can be directed to this URI to sign in with the IdP. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.createAuthUri",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1CreateAuthUriResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1CreateAuthUriRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:delete": {
            post: {
                description: "Deletes a user's account.",
                operationId: "identitytoolkit.accounts.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1DeleteAccountResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1DeleteAccountRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:issueSamlResponse": {
            post: {
                description: "Experimental",
                operationId: "identitytoolkit.accounts.issueSamlResponse",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1IssueSamlResponseResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1IssueSamlResponseRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:lookup": {
            post: {
                description: "Gets account information for all matched accounts. For an end user request, retrieves the account of the end user. For an admin request with Google OAuth 2.0 credential, retrieves one or multiple account(s) with matching criteria.",
                operationId: "identitytoolkit.accounts.lookup",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetAccountInfoResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1GetAccountInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:resetPassword": {
            post: {
                description: "Resets the password of an account either using an out-of-band code generated by sendOobCode or by specifying the email and password of the account to be modified. Can also check the purpose of an out-of-band code without consuming it.",
                operationId: "identitytoolkit.accounts.resetPassword",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1ResetPasswordResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1ResetPasswordRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:sendOobCode": {
            post: {
                description: "Sends an out-of-band confirmation code for an account. Requests from a authenticated request can optionally return a link including the OOB code instead of sending it.",
                operationId: "identitytoolkit.accounts.sendOobCode",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetOobCodeResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1GetOobCodeRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:sendVerificationCode": {
            post: {
                description: "Sends a SMS verification code for phone number sign-in. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.sendVerificationCode",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SendVerificationCodeResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SendVerificationCodeRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:signInWithCustomToken": {
            post: {
                description: "Signs in or signs up a user by exchanging a custom Auth token. Upon a successful sign-in or sign-up, a new Identity Platform ID token and refresh token are issued for the user. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.signInWithCustomToken",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithCustomTokenResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithCustomTokenRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:signInWithEmailLink": {
            post: {
                description: "Signs in or signs up a user with a out-of-band code from an email link. If a user does not exist with the given email address, a user record will be created. If the sign-in succeeds, an Identity Platform ID and refresh token are issued for the authenticated user. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.signInWithEmailLink",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithEmailLinkResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithEmailLinkRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:signInWithGameCenter": {
            post: {
                description: "Signs in or signs up a user with iOS Game Center credentials. If the sign-in succeeds, a new Identity Platform ID token and refresh token are issued for the authenticated user. The bundle ID is required in the request header as `x-ios-bundle-identifier`. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.signInWithGameCenter",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithGameCenterResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithGameCenterRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:signInWithIdp": {
            post: {
                description: 'Signs in or signs up a user using credentials from an Identity Provider (IdP). This is done by manually providing an IdP credential, or by providing the authorization response obtained via the authorization request from CreateAuthUri. If the sign-in succeeds, a new Identity Platform ID token and refresh token are issued for the authenticated user. A new Identity Platform user account will be created if the user has not previously signed in to the IdP with the same account. In addition, when the "One account per email address" setting is enabled, there should not be an existing Identity Platform user account with the same email address for a new user account to be created. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.',
                operationId: "identitytoolkit.accounts.signInWithIdp",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithIdpResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithIdpRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:signInWithPassword": {
            post: {
                description: "Signs in a user with email and password. If the sign-in succeeds, a new Identity Platform ID token and refresh token are issued for the authenticated user. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.signInWithPassword",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithPasswordResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithPasswordRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:signInWithPhoneNumber": {
            post: {
                description: "Completes a phone number authentication attempt. If a user already exists with the given phone number, an ID token is minted for that user. Otherwise, a new user is created and associated with the phone number. This method may also be used to link a phone number to an existing user. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.signInWithPhoneNumber",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithPhoneNumberResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignInWithPhoneNumberRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:signUp": {
            post: {
                description: "Signs up a new email and password user or anonymous user, or upgrades an anonymous user to email and password. For an admin request with a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control), creates a new anonymous, email and password, or phone number user. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.signUp",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignUpResponse" },
                            },
                        },
                    },
                },
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1SignUpRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:update": {
            post: {
                description: "Updates account-related information for the specified user by setting specific fields or applying action codes. Requests from administrators and end users are supported.",
                operationId: "identitytoolkit.accounts.update",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SetAccountInfoResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1SetAccountInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/accounts:verifyIosClient": {
            post: {
                description: "Verifies an iOS client is a real iOS device. If the request is valid, a receipt will be sent in the response and a secret will be sent via Apple Push Notification Service. The client should send both of them back to certain Identity Platform APIs in a later call (for example, /accounts:sendVerificationCode), in order to verify the client. The bundle ID is required in the request header as `x-ios-bundle-identifier`. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.accounts.verifyIosClient",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1VerifyIosClientResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1VerifyIosClientRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts": {
            post: {
                description: "Signs up a new email and password user or anonymous user, or upgrades an anonymous user to email and password. For an admin request with a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control), creates a new anonymous, email and password, or phone number user. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.projects.accounts",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignUpResponse" },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The project ID of the project which the user should belong to. Specifying this field requires a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control). If this is not set, the target project is inferred from the scope associated to the Bearer access token.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1SignUpRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}:createSessionCookie": {
            post: {
                description: "Creates a session cookie for the given Identity Platform ID token. The session cookie is used by the client to preserve the user's login state.",
                operationId: "identitytoolkit.projects.createSessionCookie",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1CreateSessionCookieResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the project that the account belongs to.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1CreateSessionCookieRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}:queryAccounts": {
            post: {
                description: "Looks up user accounts within a project or a tenant based on conditions in the request.",
                operationId: "identitytoolkit.projects.queryAccounts",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1QueryUserInfoResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the project to which the result is scoped.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1QueryUserInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:batchCreate": {
            post: {
                description: "Uploads multiple accounts into the Google Cloud project. If there is a problem uploading one or more of the accounts, the rest will be uploaded, and a list of the errors will be returned. To use this method requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                operationId: "identitytoolkit.projects.accounts.batchCreate",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1UploadAccountResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The Project ID of the Identity Platform project which the account belongs to.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1UploadAccountRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:batchDelete": {
            post: {
                description: "Batch deletes multiple accounts. For accounts that fail to be deleted, error info is contained in the response. The method ignores accounts that do not exist or are duplicated in the request. This method requires a Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control).",
                operationId: "identitytoolkit.projects.accounts.batchDelete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1BatchDeleteAccountsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "If `tenant_id` is specified, the ID of the Google Cloud project that the Identity Platform tenant belongs to. Otherwise, the ID of the Google Cloud project that accounts belong to.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1BatchDeleteAccountsRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:batchGet": {
            get: {
                description: "Download account information for all accounts on the project in a paginated manner. To use this method requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).. Furthermore, additional permissions are needed to get password hash, password salt, and password version from accounts; otherwise these fields are redacted.",
                operationId: "identitytoolkit.projects.accounts.batchGet",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1DownloadAccountResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "If `tenant_id` is specified, the ID of the Google Cloud project that the Identity Platform tenant belongs to. Otherwise, the ID of the Google Cloud project that the accounts belong to.",
                        required: true,
                        schema: { type: "string" },
                    },
                    { name: "delegatedProjectNumber", in: "query", schema: { type: "string" } },
                    {
                        name: "maxResults",
                        in: "query",
                        description: "The maximum number of results to return. Must be at least 1 and no greater than 1000. By default, it is 20.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "nextPageToken",
                        in: "query",
                        description: "The pagination token from the response of a previous request.",
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "query",
                        description: "The ID of the Identity Platform tenant the accounts belongs to. If not specified, accounts on the Identity Platform project are returned.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:delete": {
            post: {
                description: "Deletes a user's account.",
                operationId: "identitytoolkit.projects.accounts.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1DeleteAccountResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the project which the account belongs to. Should only be specified in authenticated requests that specify local_id of an account.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1DeleteAccountRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:lookup": {
            post: {
                description: "Gets account information for all matched accounts. For an end user request, retrieves the account of the end user. For an admin request with Google OAuth 2.0 credential, retrieves one or multiple account(s) with matching criteria.",
                operationId: "identitytoolkit.projects.accounts.lookup",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetAccountInfoResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the Google Cloud project that the account or the Identity Platform tenant specified by `tenant_id` belongs to. Should only be specified by authenticated requests bearing a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1GetAccountInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:query": {
            post: {
                description: "Looks up user accounts within a project or a tenant based on conditions in the request.",
                operationId: "identitytoolkit.projects.accounts.query",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1QueryUserInfoResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the project to which the result is scoped.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1QueryUserInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:sendOobCode": {
            post: {
                description: "Sends an out-of-band confirmation code for an account. Requests from a authenticated request can optionally return a link including the OOB code instead of sending it.",
                operationId: "identitytoolkit.projects.accounts.sendOobCode",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetOobCodeResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The Project ID of the Identity Platform project which the account belongs to. To specify this field, it requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1GetOobCodeRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/accounts:update": {
            post: {
                description: "Updates account-related information for the specified user by setting specific fields or applying action codes. Requests from administrators and end users are supported.",
                operationId: "identitytoolkit.projects.accounts.update",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SetAccountInfoResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The project ID for the project that the account belongs to. Specifying this field requires Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control). Requests from end users should pass an Identity Platform ID token instead.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1SetAccountInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts": {
            post: {
                description: "Signs up a new email and password user or anonymous user, or upgrades an anonymous user to email and password. For an admin request with a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control), creates a new anonymous, email and password, or phone number user. An [API key](https://cloud.google.com/docs/authentication/api-keys) is required in the request in order to identify the Google Cloud project.",
                operationId: "identitytoolkit.projects.tenants.accounts",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignUpResponse" },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The project ID of the project which the user should belong to. Specifying this field requires a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control). If this is not set, the target project is inferred from the scope associated to the Bearer access token.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The ID of the Identity Platform tenant to create a user under. If not set, the user will be created under the default Identity Platform project.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1SignUpRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}:createSessionCookie": {
            post: {
                description: "Creates a session cookie for the given Identity Platform ID token. The session cookie is used by the client to preserve the user's login state.",
                operationId: "identitytoolkit.projects.tenants.createSessionCookie",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1CreateSessionCookieResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the project that the account belongs to.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The tenant ID of the Identity Platform tenant the account belongs to.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1CreateSessionCookieRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:batchCreate": {
            post: {
                description: "Uploads multiple accounts into the Google Cloud project. If there is a problem uploading one or more of the accounts, the rest will be uploaded, and a list of the errors will be returned. To use this method requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                operationId: "identitytoolkit.projects.tenants.accounts.batchCreate",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1UploadAccountResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The Project ID of the Identity Platform project which the account belongs to.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The ID of the Identity Platform tenant the account belongs to.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1UploadAccountRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:batchDelete": {
            post: {
                description: "Batch deletes multiple accounts. For accounts that fail to be deleted, error info is contained in the response. The method ignores accounts that do not exist or are duplicated in the request. This method requires a Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control).",
                operationId: "identitytoolkit.projects.tenants.accounts.batchDelete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1BatchDeleteAccountsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "If `tenant_id` is specified, the ID of the Google Cloud project that the Identity Platform tenant belongs to. Otherwise, the ID of the Google Cloud project that accounts belong to.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "If the accounts belong to an Identity Platform tenant, the ID of the tenant. If the accounts belong to an default Identity Platform project, the field is not needed.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1BatchDeleteAccountsRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:batchGet": {
            get: {
                description: "Download account information for all accounts on the project in a paginated manner. To use this method requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).. Furthermore, additional permissions are needed to get password hash, password salt, and password version from accounts; otherwise these fields are redacted.",
                operationId: "identitytoolkit.projects.tenants.accounts.batchGet",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1DownloadAccountResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "If `tenant_id` is specified, the ID of the Google Cloud project that the Identity Platform tenant belongs to. Otherwise, the ID of the Google Cloud project that the accounts belong to.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The ID of the Identity Platform tenant the accounts belongs to. If not specified, accounts on the Identity Platform project are returned.",
                        required: true,
                        schema: { type: "string" },
                    },
                    { name: "delegatedProjectNumber", in: "query", schema: { type: "string" } },
                    {
                        name: "maxResults",
                        in: "query",
                        description: "The maximum number of results to return. Must be at least 1 and no greater than 1000. By default, it is 20.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "nextPageToken",
                        in: "query",
                        description: "The pagination token from the response of a previous request.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:delete": {
            post: {
                description: "Deletes a user's account.",
                operationId: "identitytoolkit.projects.tenants.accounts.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1DeleteAccountResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the project which the account belongs to. Should only be specified in authenticated requests that specify local_id of an account.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The ID of the tenant that the account belongs to, if applicable. Only require to be specified for authenticated requests bearing a Google OAuth 2.0 credential that specify local_id of an account that belongs to an Identity Platform tenant.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1DeleteAccountRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:lookup": {
            post: {
                description: "Gets account information for all matched accounts. For an end user request, retrieves the account of the end user. For an admin request with Google OAuth 2.0 credential, retrieves one or multiple account(s) with matching criteria.",
                operationId: "identitytoolkit.projects.tenants.accounts.lookup",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetAccountInfoResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the Google Cloud project that the account or the Identity Platform tenant specified by `tenant_id` belongs to. Should only be specified by authenticated requests bearing a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The ID of the tenant that the account belongs to. Should only be specified by authenticated requests from a developer.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1GetAccountInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:query": {
            post: {
                description: "Looks up user accounts within a project or a tenant based on conditions in the request.",
                operationId: "identitytoolkit.projects.tenants.accounts.query",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1QueryUserInfoResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The ID of the project to which the result is scoped.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The ID of the tenant to which the result is scoped.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1QueryUserInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:sendOobCode": {
            post: {
                description: "Sends an out-of-band confirmation code for an account. Requests from a authenticated request can optionally return a link including the OOB code instead of sending it.",
                operationId: "identitytoolkit.projects.tenants.accounts.sendOobCode",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetOobCodeResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The Project ID of the Identity Platform project which the account belongs to. To specify this field, it requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The tenant ID of the Identity Platform tenant the account belongs to.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1GetOobCodeRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts:update": {
            post: {
                description: "Updates account-related information for the specified user by setting specific fields or applying action codes. Requests from administrators and end users are supported.",
                operationId: "identitytoolkit.projects.tenants.accounts.update",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SetAccountInfoResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "targetProjectId",
                        in: "path",
                        description: "The project ID for the project that the account belongs to. Specifying this field requires Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control). Requests from end users should pass an Identity Platform ID token instead.",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "tenantId",
                        in: "path",
                        description: "The tenant ID of the Identity Platform tenant that the account belongs to. Requests from end users should pass an Identity Platform ID token rather than setting this field.",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitV1SetAccountInfoRequest",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/projects": {
            get: {
                description: "Gets a project's public Identity Toolkit configuration. (Legacy) This method also supports authenticated calls from a developer to retrieve non-public configuration.",
                operationId: "identitytoolkit.getProjects",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetProjectConfigResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "androidPackageName",
                        in: "query",
                        description: "Android package name to check against the real android package name. If this field is provided, and sha1_cert_hash is not provided, the action will throw an error if this does not match the real android package name.",
                        schema: { type: "string" },
                    },
                    {
                        name: "clientId",
                        in: "query",
                        description: "The RP OAuth client ID. If set, a check will be performed to ensure that the OAuth client is valid for the retrieved project and the request rejected with a client error if not valid.",
                        schema: { type: "string" },
                    },
                    {
                        name: "delegatedProjectNumber",
                        in: "query",
                        description: "Project Number of the delegated project request. This field should only be used as part of the Firebase V1 migration.",
                        schema: { type: "string" },
                    },
                    {
                        name: "firebaseAppId",
                        in: "query",
                        description: "The Firebase app ID, for applications that use Firebase. This can be found in the Firebase console for your project. If set, a check will be performed to ensure that the app ID is valid for the retrieved project. If not valid, the request will be rejected with a client error.",
                        schema: { type: "string" },
                    },
                    {
                        name: "iosBundleId",
                        in: "query",
                        description: "iOS bundle id to check against the real ios bundle id. If this field is provided, the action will throw an error if this does not match the real iOS bundle id.",
                        schema: { type: "string" },
                    },
                    {
                        name: "projectNumber",
                        in: "query",
                        description: "Project number of the configuration to retrieve. This field is deprecated and should not be used by new integrations.",
                        schema: { type: "string" },
                    },
                    {
                        name: "returnDynamicLink",
                        in: "query",
                        description: "Whether dynamic link should be returned.",
                        schema: { type: "boolean" },
                    },
                    {
                        name: "sha1Cert",
                        in: "query",
                        description: "SHA-1 Android application cert hash. If set, a check will be performed to ensure that the cert hash is valid for the retrieved project and android_package_name.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["v1"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/recaptchaParams": {
            get: {
                description: "Gets parameters needed for generating a reCAPTCHA challenge.",
                operationId: "identitytoolkit.getRecaptchaParams",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetRecaptchaParamResponse",
                                },
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["v1"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/sessionCookiePublicKeys": {
            get: {
                description: "Retrieves the set of public keys of the session cookie JSON Web Token (JWT) signer that can be used to validate the session cookie created through createSessionCookie.",
                operationId: "identitytoolkit.getSessionCookiePublicKeys",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetSessionCookiePublicKeysResponse",
                                },
                            },
                        },
                    },
                },
                tags: ["v1"],
                security: [{ apiKeyQuery: [] }, { apiKeyHeader: [] }],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/accounts/mfaEnrollment:finalize": {
            post: {
                description: "Finishes enrolling a second factor for the user.",
                operationId: "identitytoolkit.accounts.mfaEnrollment.finalize",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaEnrollmentResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaEnrollmentRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/accounts/mfaEnrollment:start": {
            post: {
                description: "Step one of the MFA enrollment process. In SMS case, this sends an SMS verification code to the user.",
                operationId: "identitytoolkit.accounts.mfaEnrollment.start",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaEnrollmentResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaEnrollmentRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/accounts/mfaEnrollment:withdraw": {
            post: {
                description: "Revokes one second factor from the enrolled second factors for an account.",
                operationId: "identitytoolkit.accounts.mfaEnrollment.withdraw",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2WithdrawMfaResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2WithdrawMfaRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/accounts/mfaSignIn:finalize": {
            post: {
                description: "Verifies the MFA challenge and performs sign-in",
                operationId: "identitytoolkit.accounts.mfaSignIn.finalize",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaSignInResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaSignInRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/accounts/mfaSignIn:start": {
            post: {
                description: "Sends the MFA challenge",
                operationId: "identitytoolkit.accounts.mfaSignIn.start",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaSignInResponse",
                                },
                            },
                        },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaSignInRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["accounts"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/defaultSupportedIdps": {
            get: {
                description: "List all default supported Idps.",
                operationId: "identitytoolkit.defaultSupportedIdps.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListDefaultSupportedIdpsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of items to return.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The next_page_token value returned from a previous List request, if any.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["defaultSupportedIdps"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/config": {
            get: {
                description: "Retrieve an Identity Toolkit project configuration.",
                operationId: "identitytoolkit.projects.getConfig",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Config" },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update an Identity Toolkit project configuration.",
                operationId: "identitytoolkit.projects.updateConfig",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Config" },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "The update mask applies to the resource. Fields set in the config but not included in this update mask will be ignored. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    content: {
                        "application/json": {
                            schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Config" },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/defaultSupportedIdpConfigs": {
            post: {
                description: "Create a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.defaultSupportedIdpConfigs.create",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "idpId",
                        in: "query",
                        description: "The id of the Idp to create a config for. Call ListDefaultSupportedIdps for list of all default supported Idps.",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "List all default supported Idp configurations for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.defaultSupportedIdpConfigs.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListDefaultSupportedIdpConfigsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of items to return.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The next_page_token value returned from a previous List request, if any.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/defaultSupportedIdpConfigs/{defaultSupportedIdpConfigsId}": {
            delete: {
                description: "Delete a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.defaultSupportedIdpConfigs.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleProtobufEmpty" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "defaultSupportedIdpConfigsId",
                        in: "path",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "Retrieve a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.defaultSupportedIdpConfigs.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "defaultSupportedIdpConfigsId",
                        in: "path",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.defaultSupportedIdpConfigs.patch",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "defaultSupportedIdpConfigsId",
                        in: "path",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "The update mask applies to the resource. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/identityPlatform:initializeAuth": {
            post: {
                description: "Initialize Identity Platform for a Cloud project. Identity Platform is an end-to-end authentication system for third-party users to access your apps and services. These could include mobile/web apps, games, APIs and beyond. This is the publicly available variant of EnableIdentityPlatform that is only available to billing-enabled projects.",
                operationId: "identitytoolkit.projects.identityPlatform.initializeAuth",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InitializeIdentityPlatformResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                ],
                requestBody: {
                    content: {
                        "application/json": {
                            schema: {
                                $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InitializeIdentityPlatformRequest",
                            },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/inboundSamlConfigs": {
            post: {
                description: "Create an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.inboundSamlConfigs.create",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "inboundSamlConfigId",
                        in: "query",
                        description: "The id to use for this config.",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "List all inbound SAML configurations for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.inboundSamlConfigs.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListInboundSamlConfigsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of items to return.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The next_page_token value returned from a previous List request, if any.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/inboundSamlConfigs/{inboundSamlConfigsId}": {
            delete: {
                description: "Delete an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.inboundSamlConfigs.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleProtobufEmpty" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "inboundSamlConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "Retrieve an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.inboundSamlConfigs.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "inboundSamlConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.inboundSamlConfigs.patch",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "inboundSamlConfigsId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "The update mask applies to the resource. Empty update mask will result in updating nothing. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/oauthIdpConfigs": {
            post: {
                description: "Create an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.oauthIdpConfigs.create",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "oauthIdpConfigId",
                        in: "query",
                        description: "The id to use for this config.",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "List all Oidc Idp configurations for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.oauthIdpConfigs.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListOAuthIdpConfigsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of items to return.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The next_page_token value returned from a previous List request, if any.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/oauthIdpConfigs/{oauthIdpConfigsId}": {
            delete: {
                description: "Delete an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.oauthIdpConfigs.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleProtobufEmpty" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "oauthIdpConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "Retrieve an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.oauthIdpConfigs.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "oauthIdpConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.oauthIdpConfigs.patch",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "oauthIdpConfigsId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "The update mask applies to the resource. Empty update mask will result in updating nothing. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants": {
            post: {
                description: "Create a tenant. Requires write permission on the Agent project.",
                operationId: "identitytoolkit.projects.tenants.create",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Tenant" },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                ],
                requestBody: { $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2Tenant" },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "List tenants under the given agent project. Requires read permission on the Agent project.",
                operationId: "identitytoolkit.projects.tenants.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListTenantsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of results to return, capped at 1000. If not specified, the default value is 20.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The pagination token from the response of a previous request.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}": {
            delete: {
                description: "Delete a tenant. Requires write permission on the Agent project.",
                operationId: "identitytoolkit.projects.tenants.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleProtobufEmpty" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "Get a tenant. Requires read permission on the Tenant resource.",
                operationId: "identitytoolkit.projects.tenants.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Tenant" },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update a tenant. Requires write permission on the Tenant resource.",
                operationId: "identitytoolkit.projects.tenants.patch",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Tenant" },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "If provided, only update fields set in the update mask. Otherwise, all settable fields will be updated. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: { $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2Tenant" },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}:getIamPolicy": {
            post: {
                description: "Gets the access control policy for a resource. An error is returned if the resource does not exist. An empty policy is returned if the resource exists but does not have a policy set on it. Caller must have the right Google IAM permission on the resource.",
                operationId: "identitytoolkit.projects.tenants.getIamPolicy",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleIamV1Policy" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                ],
                requestBody: {
                    content: {
                        "application/json": {
                            schema: { $ref: "#/components/schemas/GoogleIamV1GetIamPolicyRequest" },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}:setIamPolicy": {
            post: {
                description: "Sets the access control policy for a resource. If the policy exists, it is replaced. Caller must have the right Google IAM permission on the resource.",
                operationId: "identitytoolkit.projects.tenants.setIamPolicy",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleIamV1Policy" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                ],
                requestBody: {
                    content: {
                        "application/json": {
                            schema: { $ref: "#/components/schemas/GoogleIamV1SetIamPolicyRequest" },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}:testIamPermissions": {
            post: {
                description: "Returns the caller's permissions on a resource. An error is returned if the resource does not exist. A caller is not required to have Google IAM permission to make this request.",
                operationId: "identitytoolkit.projects.tenants.testIamPermissions",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: { $ref: "#/components/schemas/GoogleIamV1TestIamPermissionsResponse" },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                ],
                requestBody: {
                    content: {
                        "application/json": {
                            schema: { $ref: "#/components/schemas/GoogleIamV1TestIamPermissionsRequest" },
                        },
                    },
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}/defaultSupportedIdpConfigs": {
            post: {
                description: "Create a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.defaultSupportedIdpConfigs.create",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "idpId",
                        in: "query",
                        description: "The id of the Idp to create a config for. Call ListDefaultSupportedIdps for list of all default supported Idps.",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "List all default supported Idp configurations for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.defaultSupportedIdpConfigs.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListDefaultSupportedIdpConfigsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of items to return.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The next_page_token value returned from a previous List request, if any.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}/defaultSupportedIdpConfigs/{defaultSupportedIdpConfigsId}": {
            delete: {
                description: "Delete a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.defaultSupportedIdpConfigs.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleProtobufEmpty" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "defaultSupportedIdpConfigsId",
                        in: "path",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "Retrieve a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.defaultSupportedIdpConfigs.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "defaultSupportedIdpConfigsId",
                        in: "path",
                        required: true,
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update a default supported Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.defaultSupportedIdpConfigs.patch",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "defaultSupportedIdpConfigsId",
                        in: "path",
                        required: true,
                        schema: { type: "string" },
                    },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "The update mask applies to the resource. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}/inboundSamlConfigs": {
            post: {
                description: "Create an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.inboundSamlConfigs.create",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "inboundSamlConfigId",
                        in: "query",
                        description: "The id to use for this config.",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "List all inbound SAML configurations for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.inboundSamlConfigs.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListInboundSamlConfigsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of items to return.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The next_page_token value returned from a previous List request, if any.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}/inboundSamlConfigs/{inboundSamlConfigsId}": {
            delete: {
                description: "Delete an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.inboundSamlConfigs.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleProtobufEmpty" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    { name: "inboundSamlConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "Retrieve an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.inboundSamlConfigs.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    { name: "inboundSamlConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update an inbound SAML configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.inboundSamlConfigs.patch",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    { name: "inboundSamlConfigsId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "The update mask applies to the resource. Empty update mask will result in updating nothing. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}/oauthIdpConfigs": {
            post: {
                description: "Create an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.oauthIdpConfigs.create",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "oauthIdpConfigId",
                        in: "query",
                        description: "The id to use for this config.",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "List all Oidc Idp configurations for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.oauthIdpConfigs.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ListOAuthIdpConfigsResponse",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "pageSize",
                        in: "query",
                        description: "The maximum number of items to return.",
                        schema: { type: "integer" },
                    },
                    {
                        name: "pageToken",
                        in: "query",
                        description: "The next_page_token value returned from a previous List request, if any.",
                        schema: { type: "string" },
                    },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v2/projects/{targetProjectId}/tenants/{tenantId}/oauthIdpConfigs/{oauthIdpConfigsId}": {
            delete: {
                description: "Delete an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.oauthIdpConfigs.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GoogleProtobufEmpty" } } },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    { name: "oauthIdpConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            get: {
                description: "Retrieve an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.oauthIdpConfigs.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    { name: "oauthIdpConfigsId", in: "path", required: true, schema: { type: "string" } },
                ],
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            patch: {
                description: "Update an Oidc Idp configuration for an Identity Toolkit project.",
                operationId: "identitytoolkit.projects.tenants.oauthIdpConfigs.patch",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "*/*": {
                                schema: {
                                    $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                                },
                            },
                        },
                    },
                },
                parameters: [
                    { name: "targetProjectId", in: "path", required: true, schema: { type: "string" } },
                    { name: "tenantId", in: "path", required: true, schema: { type: "string" } },
                    { name: "oauthIdpConfigsId", in: "path", required: true, schema: { type: "string" } },
                    {
                        name: "updateMask",
                        in: "query",
                        description: "The update mask applies to the resource. Empty update mask will result in updating nothing. For the `FieldMask` definition, see https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask",
                        schema: { type: "string" },
                    },
                ],
                requestBody: {
                    $ref: "#/components/requestBodies/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                },
                security: [
                    { Oauth2: ["https://www.googleapis.com/auth/cloud-platform"] },
                    { Oauth2: ["https://www.googleapis.com/auth/firebase"] },
                    { apiKeyQuery: [] },
                    { apiKeyHeader: [] },
                ],
                tags: ["projects"],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
        },
        "/v1/token": {
            post: {
                description: "The Token Service API lets you exchange either an ID token or a refresh token for an access token and a new refresh token. You can use the access token to securely call APIs that require user authorization.",
                operationId: "securetoken.token",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "*/*": { schema: { $ref: "#/components/schemas/GrantTokenResponse" } } },
                    },
                },
                requestBody: {
                    content: {
                        "application/json": { schema: { $ref: "#/components/schemas/GrantTokenRequest" } },
                        "application/x-www-form-urlencoded": {
                            schema: { $ref: "#/components/schemas/GrantTokenRequest" },
                        },
                    },
                },
                tags: ["secureToken"],
                security: [{ apiKeyQuery: [] }, { apiKeyHeader: [] }],
            },
            parameters: [
                { $ref: "#/components/parameters/access_token" },
                { $ref: "#/components/parameters/alt" },
                { $ref: "#/components/parameters/callback" },
                { $ref: "#/components/parameters/fields" },
                { $ref: "#/components/parameters/oauth_token" },
                { $ref: "#/components/parameters/prettyPrint" },
                { $ref: "#/components/parameters/quotaUser" },
                { $ref: "#/components/parameters/uploadType" },
                { $ref: "#/components/parameters/upload_protocol" },
            ],
            servers: [{ url: "https://securetoken.googleapis.com" }],
        },
        "/emulator/v1/projects/{targetProjectId}/accounts": {
            parameters: [
                {
                    name: "targetProjectId",
                    in: "path",
                    description: "The ID of the Google Cloud project that the accounts belong to.",
                    required: true,
                    schema: { type: "string" },
                },
            ],
            servers: [{ url: "" }],
            delete: {
                description: "Remove all accounts in the project, regardless of state.",
                operationId: "emulator.projects.accounts.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "application/json": { schema: { type: "object" } } },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
        },
        "/emulator/v1/projects/{targetProjectId}/tenants/{tenantId}/accounts": {
            parameters: [
                {
                    name: "targetProjectId",
                    in: "path",
                    description: "The ID of the Google Cloud project that the accounts belong to.",
                    required: true,
                    schema: { type: "string" },
                },
                {
                    name: "tenantId",
                    in: "path",
                    description: "The ID of the Identity Platform tenant the accounts belongs to. If not specified, accounts on the Identity Platform project are returned.",
                    required: true,
                    schema: { type: "string" },
                },
            ],
            servers: [{ url: "" }],
            delete: {
                description: "Remove all accounts in the project, regardless of state.",
                operationId: "emulator.projects.accounts.delete",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: { "application/json": { schema: { type: "object" } } },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
        },
        "/emulator/v1/projects/{targetProjectId}/config": {
            parameters: [
                {
                    name: "targetProjectId",
                    in: "path",
                    description: "The ID of the Google Cloud project that the config belongs to.",
                    required: true,
                    schema: { type: "string" },
                },
            ],
            servers: [{ url: "" }],
            get: {
                description: "Get emulator-specific configuration for the project.",
                operationId: "emulator.projects.config.get",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "application/json": {
                                schema: { $ref: "#/components/schemas/EmulatorV1ProjectsConfig" },
                            },
                        },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
            patch: {
                description: "Update emulator-specific configuration for the project.",
                operationId: "emulator.projects.config.update",
                requestBody: {
                    content: {
                        "application/json": {
                            schema: { $ref: "#/components/schemas/EmulatorV1ProjectsConfig" },
                        },
                    },
                },
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "application/json": {
                                schema: { $ref: "#/components/schemas/EmulatorV1ProjectsConfig" },
                            },
                        },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
        },
        "/emulator/v1/projects/{targetProjectId}/oobCodes": {
            parameters: [
                {
                    name: "targetProjectId",
                    in: "path",
                    description: "The ID of the Google Cloud project that the confirmation codes belongs to.",
                    required: true,
                    schema: { type: "string" },
                },
            ],
            servers: [{ url: "" }],
            get: {
                description: "List all pending confirmation codes for the project.",
                operationId: "emulator.projects.oobCodes.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "application/json": {
                                schema: { $ref: "#/components/schemas/EmulatorV1ProjectsOobCodes" },
                            },
                        },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
        },
        "/emulator/v1/projects/{targetProjectId}/tenants/{tenantId}/oobCodes": {
            parameters: [
                {
                    name: "targetProjectId",
                    in: "path",
                    description: "The ID of the Google Cloud project that the confirmation codes belongs to.",
                    required: true,
                    schema: { type: "string" },
                },
                {
                    name: "tenantId",
                    in: "path",
                    description: "The ID of the Identity Platform tenant the accounts belongs to. If not specified, accounts on the Identity Platform project are returned.",
                    required: true,
                    schema: { type: "string" },
                },
            ],
            servers: [{ url: "" }],
            get: {
                description: "List all pending confirmation codes for the project.",
                operationId: "emulator.projects.oobCodes.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "application/json": {
                                schema: { $ref: "#/components/schemas/EmulatorV1ProjectsOobCodes" },
                            },
                        },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
        },
        "/emulator/v1/projects/{targetProjectId}/verificationCodes": {
            parameters: [
                {
                    name: "targetProjectId",
                    in: "path",
                    description: "The ID of the Google Cloud project that the verification codes belongs to.",
                    required: true,
                    schema: { type: "string" },
                },
            ],
            servers: [{ url: "" }],
            get: {
                description: "List all pending phone verification codes for the project.",
                operationId: "emulator.projects.verificationCodes.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "application/json": {
                                schema: { $ref: "#/components/schemas/EmulatorV1ProjectsOobCodes" },
                            },
                        },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
        },
        "/emulator/v1/projects/{targetProjectId}/tenants/{tenantId}/verificationCodes": {
            parameters: [
                {
                    name: "targetProjectId",
                    in: "path",
                    description: "The ID of the Google Cloud project that the verification codes belongs to.",
                    required: true,
                    schema: { type: "string" },
                },
                {
                    name: "tenantId",
                    in: "path",
                    description: "The ID of the Identity Platform tenant the accounts belongs to. If not specified, accounts on the Identity Platform project are returned.",
                    required: true,
                    schema: { type: "string" },
                },
            ],
            servers: [{ url: "" }],
            get: {
                description: "List all pending phone verification codes for the project.",
                operationId: "emulator.projects.verificationCodes.list",
                responses: {
                    "200": {
                        description: "Successful response",
                        content: {
                            "application/json": {
                                schema: { $ref: "#/components/schemas/EmulatorV1ProjectsOobCodes" },
                            },
                        },
                    },
                },
                security: [],
                tags: ["emulator"],
            },
        },
    },
    components: {
        schemas: {
            GoogleCloudIdentitytoolkitV1Argon2Parameters: {
                description: "The parameters for Argon2 hashing algorithm.",
                properties: {
                    associatedData: {
                        description: "The additional associated data, if provided, is appended to the hash value to provide an additional layer of security. A base64-encoded string if specified via JSON.",
                        format: "byte",
                        type: "string",
                    },
                    hashLengthBytes: {
                        description: "Required. The desired hash length in bytes. Minimum is 4 and maximum is 1024.",
                        format: "int32",
                        type: "integer",
                    },
                    hashType: {
                        description: "Required. Must not be HASH_TYPE_UNSPECIFIED.",
                        enum: ["HASH_TYPE_UNSPECIFIED", "ARGON2_D", "ARGON2_ID", "ARGON2_I"],
                        type: "string",
                    },
                    iterations: {
                        description: "Required. The number of iterations to perform. Minimum is 1, maximum is 16.",
                        format: "int32",
                        type: "integer",
                    },
                    memoryCostKib: {
                        description: "Required. The memory cost in kibibytes. Maximum is 32768.",
                        format: "int32",
                        type: "integer",
                    },
                    parallelism: {
                        description: "Required. The degree of parallelism, also called threads or lanes. Minimum is 1, maximum is 16.",
                        format: "int32",
                        type: "integer",
                    },
                    version: {
                        description: "The version of the Argon2 algorithm. This defaults to VERSION_13 if not specified.",
                        enum: ["VERSION_UNSPECIFIED", "VERSION_10", "VERSION_13"],
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1AutoRetrievalInfo: {
                description: "The information required to auto-retrieve an SMS.",
                properties: {
                    appSignatureHash: {
                        description: "The Android app's signature hash for Google Play Service's SMS Retriever API.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1BatchDeleteAccountsRequest: {
                description: "Request message for BatchDeleteAccounts.",
                properties: {
                    force: {
                        description: "Whether to force deleting accounts that are not in disabled state. If false, only disabled accounts will be deleted, and accounts that are not disabled will be added to the `errors`.",
                        type: "boolean",
                    },
                    localIds: {
                        description: "Required. List of user IDs to be deleted.",
                        items: { type: "string" },
                        type: "array",
                    },
                    tenantId: {
                        description: "If the accounts belong to an Identity Platform tenant, the ID of the tenant. If the accounts belong to an default Identity Platform project, the field is not needed.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1BatchDeleteAccountsResponse: {
                description: "Response message to BatchDeleteAccounts.",
                properties: {
                    errors: {
                        description: "Detailed error info for accounts that cannot be deleted.",
                        items: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1BatchDeleteErrorInfo",
                        },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1BatchDeleteErrorInfo: {
                description: "Error info for account failed to be deleted.",
                properties: {
                    index: {
                        description: "The index of the errored item in the original local_ids field.",
                        format: "int32",
                        type: "integer",
                    },
                    localId: { description: "The corresponding user ID.", type: "string" },
                    message: { description: "Detailed error message.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1CreateAuthUriRequest: {
                description: "Request message for CreateAuthUri.",
                properties: {
                    appId: { type: "string" },
                    authFlowType: {
                        description: "Used for the Google provider. The type of the authentication flow to be used. If present, this should be `CODE_FLOW` to specify the authorization code flow. Otherwise, the default ID Token flow will be used.",
                        type: "string",
                    },
                    context: {
                        description: "An opaque string used to maintain contextual information between the authentication request and the callback from the IdP.",
                        type: "string",
                    },
                    continueUri: {
                        description: "A valid URL for the IdP to redirect the user back to. The URL cannot contain fragments or the reserved `state` query parameter.",
                        type: "string",
                    },
                    customParameter: {
                        additionalProperties: { type: "string" },
                        description: "Additional customized query parameters to be added to the authorization URI. The following parameters are reserved and cannot be added: `client_id`, `response_type`, `scope`, `redirect_uri`, `state`. For the Microsoft provider, the Azure AD tenant to sign-in to can be specified in the `tenant` custom parameter.",
                        type: "object",
                    },
                    hostedDomain: {
                        description: "Used for the Google provider. The G Suite hosted domain of the user in order to restrict sign-in to users at that domain.",
                        type: "string",
                    },
                    identifier: {
                        description: "The email identifier of the user account to fetch associated providers for. At least one of the fields `identifier` and `provider_id` must be set. The length of the email address should be less than 256 characters and in the format of `name@domain.tld`. The email address should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec production.",
                        type: "string",
                    },
                    oauthConsumerKey: { type: "string" },
                    oauthScope: {
                        description: "Additional space-delimited OAuth 2.0 scopes specifying the scope of the authentication request with the IdP. Used for OAuth 2.0 IdPs. For the Google provider, the authorization code flow will be used if this field is set.",
                        type: "string",
                    },
                    openidRealm: { type: "string" },
                    otaApp: { type: "string" },
                    providerId: {
                        description: "The provider ID of the IdP for the user to sign in with. This should be a provider ID enabled for sign-in, which is either from the list of [default supported IdPs](https://cloud.google.com/identity-platform/docs/reference/rest/v2/defaultSupportedIdps/list), or of the format `oidc.*` or `saml.*`. Some examples are `google.com`, `facebook.com`, `oidc.testapp`, and `saml.testapp`. At least one of the fields `identifier` and `provider_id` must be set.",
                        type: "string",
                    },
                    sessionId: {
                        description: "A session ID that can be verified against in SignInWithIdp to prevent session fixation attacks. If absent, a random string will be generated and returned as the session ID.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant to create an authorization URI or lookup an email identifier for. If not set, the operation will be performed in the default Identity Platform instance in the project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1CreateAuthUriResponse: {
                description: "Response message for CreateAuthUri.",
                properties: {
                    allProviders: { items: { type: "string" }, type: "array" },
                    authUri: {
                        description: "The authorization URI for the requested provider. Present only when a provider ID is set in the request.",
                        type: "string",
                    },
                    captchaRequired: {
                        description: "Whether a CAPTCHA is needed because there have been too many failed login attempts by the user. Present only when a registered email identifier is set in the request.",
                        type: "boolean",
                    },
                    forExistingProvider: {
                        description: "Whether the user has previously signed in with the provider ID in the request. Present only when a registered email identifier is set in the request.",
                        type: "boolean",
                    },
                    kind: { type: "string" },
                    providerId: {
                        description: "The provider ID from the request, if provided.",
                        type: "string",
                    },
                    registered: {
                        description: "Whether the email identifier represents an existing account. Present only when an email identifier is set in the request.",
                        type: "boolean",
                    },
                    sessionId: {
                        description: "The session ID from the request, or a random string generated by CreateAuthUri if absent. It is used to prevent session fixation attacks.",
                        type: "string",
                    },
                    signinMethods: {
                        description: "The list of sign-in methods that the user has previously used. Each element is one of `password`, `emailLink`, or the provider ID of an IdP. Present only when a registered email identifier is set in the request.",
                        items: { type: "string" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1CreateSessionCookieRequest: {
                description: "Request message for CreateSessionCookie.",
                properties: {
                    idToken: { description: "Required. A valid Identity Platform ID token.", type: "string" },
                    tenantId: {
                        description: "The tenant ID of the Identity Platform tenant the account belongs to.",
                        type: "string",
                    },
                    validDuration: {
                        description: "The number of seconds until the session cookie expires. Specify a duration in seconds, between five minutes and fourteen days, inclusively.",
                        format: "int64",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1CreateSessionCookieResponse: {
                description: "Response message for CreateSessionCookie.",
                properties: {
                    sessionCookie: {
                        description: "The session cookie that has been created from the Identity Platform ID token specified in the request. It is in the form of a JSON Web Token (JWT). Always present.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1DeleteAccountRequest: {
                description: "Request message for DeleteAccount.",
                properties: {
                    delegatedProjectNumber: { format: "int64", type: "string" },
                    idToken: {
                        description: "The Identity Platform ID token of the account to delete. Require to be specified for requests from end users that lack Google OAuth 2.0 credential. Authenticated requests bearing a Google OAuth2 credential with proper permissions may pass local_id to specify the account to delete alternatively.",
                        type: "string",
                    },
                    localId: {
                        description: "The ID of user account to delete. Specifying this field requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control). Requests from users lacking the credential should pass an ID token instead.",
                        type: "string",
                    },
                    targetProjectId: {
                        description: "The ID of the project which the account belongs to. Should only be specified in authenticated requests that specify local_id of an account.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the tenant that the account belongs to, if applicable. Only require to be specified for authenticated requests bearing a Google OAuth 2.0 credential that specify local_id of an account that belongs to an Identity Platform tenant.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1DeleteAccountResponse: {
                description: "Response message for DeleteAccount.",
                properties: { kind: { type: "string" } },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1DownloadAccountResponse: {
                description: "Response message for DownloadAccount.",
                properties: {
                    kind: { type: "string" },
                    nextPageToken: {
                        description: "If there are more accounts to be downloaded, a token that can be passed back to DownloadAccount to get more accounts. Otherwise, this is blank.",
                        type: "string",
                    },
                    users: {
                        description: "All accounts belonging to the project/tenant limited by max_results in the request.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1UserInfo" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1EmailTemplate: {
                description: "Email template",
                properties: {
                    body: { description: "Email body", type: "string" },
                    customized: {
                        description: "Whether the body or subject of the email is customized.",
                        type: "boolean",
                    },
                    disabled: {
                        description: "Whether the template is disabled. If true, a default template will be used.",
                        type: "boolean",
                    },
                    format: {
                        description: "Email body format",
                        enum: ["EMAIL_BODY_FORMAT_UNSPECIFIED", "PLAINTEXT", "HTML"],
                        type: "string",
                    },
                    from: { description: "From address of the email", type: "string" },
                    fromDisplayName: { description: "From display name", type: "string" },
                    fromLocalPart: { description: "Local part of From address", type: "string" },
                    locale: {
                        description: "Value is in III language code format (e.g. \"zh-CN\", \"es\"). Both '-' and '_' separators are accepted.",
                        type: "string",
                    },
                    replyTo: { description: "Reply-to address", type: "string" },
                    subject: { description: "Subject of the email", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1ErrorInfo: {
                description: "Error information explaining why an account cannot be uploaded. batch upload.",
                properties: {
                    index: {
                        description: "The index of the item, range is [0, request.size - 1]",
                        format: "int32",
                        type: "integer",
                    },
                    message: { description: "Detailed error message", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1FederatedUserIdentifier: {
                description: "Federated user identifier at an Identity Provider.",
                properties: {
                    providerId: {
                        description: "The ID of supported identity providers. This should be a provider ID enabled for sign-in, which is either from the list of [default supported IdPs](https://cloud.google.com/identity-platform/docs/reference/rest/v2/defaultSupportedIdps/list), or of the format `oidc.*` or `saml.*`. Some examples are `google.com`, `facebook.com`, `oidc.testapp`, and `saml.testapp`.",
                        type: "string",
                    },
                    rawId: {
                        description: "The user ID of the account at the third-party Identity Provider specified by `provider_id`.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1GetAccountInfoRequest: {
                description: "Request message for GetAccountInfo.",
                properties: {
                    delegatedProjectNumber: { format: "int64", type: "string" },
                    email: {
                        description: "The email address of one or more accounts to fetch. The length of email should be less than 256 characters and in the format of `name@domain.tld`. The email should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec production. Should only be specified by authenticated requests from a developer.",
                        items: { type: "string" },
                        type: "array",
                    },
                    federatedUserId: {
                        description: "The federated user identifier of one or more accounts to fetch. Should only be specified by authenticated requests bearing a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        items: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1FederatedUserIdentifier",
                        },
                        type: "array",
                    },
                    idToken: {
                        description: "The Identity Platform ID token of the account to fetch. Require to be specified for requests from end users.",
                        type: "string",
                    },
                    initialEmail: {
                        description: "The initial email of one or more accounts to fetch. The length of email should be less than 256 characters and in the format of `name@domain.tld`. The email should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec production. Should only be specified by authenticated requests from a developer.",
                        items: { type: "string" },
                        type: "array",
                    },
                    localId: {
                        description: "The ID of one or more accounts to fetch. Should only be specified by authenticated requests bearing a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        items: { type: "string" },
                        type: "array",
                    },
                    phoneNumber: {
                        description: "The phone number of one or more accounts to fetch. Should only be specified by authenticated requests from a developer and should be in E.164 format, for example, +15555555555.",
                        items: { type: "string" },
                        type: "array",
                    },
                    targetProjectId: {
                        description: "The ID of the Google Cloud project that the account or the Identity Platform tenant specified by `tenant_id` belongs to. Should only be specified by authenticated requests bearing a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the tenant that the account belongs to. Should only be specified by authenticated requests from a developer.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1GetAccountInfoResponse: {
                description: "Response message for GetAccountInfo.",
                properties: {
                    kind: { type: "string" },
                    users: {
                        description: "The information of specific user account(s) matching the parameters in the request.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1UserInfo" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1GetOobCodeRequest: {
                description: "Request message for GetOobCode.",
                properties: {
                    androidInstallApp: {
                        description: "If an associated android app can handle the OOB code, whether or not to install the android app on the device where the link is opened if the app is not already installed.",
                        type: "boolean",
                    },
                    androidMinimumVersion: {
                        description: "If an associated android app can handle the OOB code, the minimum version of the app. If the version on the device is lower than this version then the user is taken to Google Play Store to upgrade the app.",
                        type: "string",
                    },
                    androidPackageName: {
                        description: "If an associated android app can handle the OOB code, the Android package name of the android app that will handle the callback when this OOB code is used. This will allow the correct app to open if it is already installed, or allow Google Play Store to open to the correct app if it is not yet installed.",
                        type: "string",
                    },
                    canHandleCodeInApp: {
                        description: "When set to true, the OOB code link will be be sent as a Universal Link or an Android App Link and will be opened by the corresponding app if installed. If not set, or set to false, the OOB code will be sent to the web widget first and then on continue will redirect to the app if installed.",
                        type: "boolean",
                    },
                    captchaResp: {
                        description: "For a PASSWORD_RESET request, a reCaptcha response is required when the system detects possible abuse activity. In those cases, this is the response from the reCaptcha challenge used to verify the caller.",
                        type: "string",
                    },
                    challenge: { type: "string" },
                    continueUrl: {
                        description: "The Url to continue after user clicks the link sent in email. This is the url that will allow the web widget to handle the OOB code.",
                        type: "string",
                    },
                    dynamicLinkDomain: {
                        description: "In order to ensure that the url used can be easily opened up in iOS or android, we create a [Firebase Dynamic Link](https://firebase.google.com/docs/dynamic-links). Most Identity Platform projects will only have one Dynamic Link domain enabled, and can leave this field blank. This field contains a specified Dynamic Link domain for projects that have multiple enabled.",
                        type: "string",
                    },
                    email: {
                        description: "The account's email address to send the OOB code to, and generally the email address of the account that needs to be updated. Required for PASSWORD_RESET, EMAIL_SIGNIN, and VERIFY_EMAIL. Only required for VERIFY_AND_CHANGE_EMAIL requests when return_oob_link is set to true. In this case, it is the original email of the user.",
                        type: "string",
                    },
                    iOSAppStoreId: {
                        description: "If an associated iOS app can handle the OOB code, the App Store id of this app. This will allow App Store to open to the correct app if the app is not yet installed.",
                        type: "string",
                    },
                    iOSBundleId: {
                        description: "If an associated iOS app can handle the OOB code, the iOS bundle id of this app. This will allow the correct app to open if it is already installed.",
                        type: "string",
                    },
                    idToken: {
                        description: "An ID token for the account. It is required for VERIFY_AND_CHANGE_EMAIL and VERIFY_EMAIL requests unless return_oob_link is set to true.",
                        type: "string",
                    },
                    newEmail: {
                        description: "The email address the account is being updated to. Required only for VERIFY_AND_CHANGE_EMAIL requests.",
                        type: "string",
                    },
                    requestType: {
                        description: "Required. The type of out-of-band (OOB) code to send. Depending on this value, other fields in this request will be required and/or have different meanings. There are 4 different OOB codes that can be sent: * PASSWORD_RESET * EMAIL_SIGNIN * VERIFY_EMAIL * VERIFY_AND_CHANGE_EMAIL",
                        enum: [
                            "OOB_REQ_TYPE_UNSPECIFIED",
                            "PASSWORD_RESET",
                            "OLD_EMAIL_AGREE",
                            "NEW_EMAIL_ACCEPT",
                            "VERIFY_EMAIL",
                            "RECOVER_EMAIL",
                            "EMAIL_SIGNIN",
                            "VERIFY_AND_CHANGE_EMAIL",
                            "REVERT_SECOND_FACTOR_ADDITION",
                        ],
                        type: "string",
                    },
                    returnOobLink: {
                        description: "Whether the confirmation link containing the OOB code should be returned in the response (no email is sent). Used when a developer wants to construct the email template and send it on their own. By default this is false; to specify this field, and to set it to true, it requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control)",
                        type: "boolean",
                    },
                    targetProjectId: {
                        description: "The Project ID of the Identity Platform project which the account belongs to. To specify this field, it requires a Google OAuth 2.0 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "string",
                    },
                    tenantId: {
                        description: "The tenant ID of the Identity Platform tenant the account belongs to.",
                        type: "string",
                    },
                    userIp: {
                        description: "The IP address of the caller. Required only for PASSWORD_RESET requests.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1GetOobCodeResponse: {
                description: "Response message for GetOobCode.",
                properties: {
                    email: {
                        description: "If return_oob_link is false in the request, the email address the verification was sent to.",
                        type: "string",
                    },
                    kind: { type: "string" },
                    oobCode: {
                        description: "If return_oob_link is true in the request, the OOB code to send.",
                        type: "string",
                    },
                    oobLink: {
                        description: "If return_oob_link is true in the request, the OOB link to be sent to the user. This returns the constructed link including [Firebase Dynamic Link](https://firebase.google.com/docs/dynamic-links) related parameters.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1GetProjectConfigResponse: {
                description: "Response message for GetProjectConfig.",
                properties: {
                    allowPasswordUser: {
                        description: "Whether to allow password account sign up. This field is only returned for authenticated calls from a developer.",
                        type: "boolean",
                    },
                    apiKey: {
                        description: "Google Cloud API key. This field is only returned for authenticated calls from a developer.",
                        type: "string",
                    },
                    authorizedDomains: {
                        description: "Authorized domains for widget redirect.",
                        items: { type: "string" },
                        type: "array",
                    },
                    changeEmailTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1EmailTemplate",
                    },
                    dynamicLinksDomain: {
                        description: "The Firebase Dynamic Links domain used to construct links for redirects to native apps.",
                        type: "string",
                    },
                    enableAnonymousUser: {
                        description: "Whether anonymous user is enabled. This field is only returned for authenticated calls from a developer.",
                        type: "boolean",
                    },
                    idpConfig: {
                        description: "OAuth2 provider config. This field is only returned for authenticated calls from a developer.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1IdpConfig" },
                        type: "array",
                    },
                    legacyResetPasswordTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1EmailTemplate",
                    },
                    projectId: {
                        description: "The project id of the retrieved configuration.",
                        type: "string",
                    },
                    resetPasswordTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1EmailTemplate",
                    },
                    revertSecondFactorAdditionTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1EmailTemplate",
                    },
                    useEmailSending: {
                        description: "Whether to use email sending. This field is only returned for authenticated calls from a developer.",
                        type: "boolean",
                    },
                    verifyEmailTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1EmailTemplate",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1GetRecaptchaParamResponse: {
                description: "Response message for GetRecaptchaParam.",
                properties: {
                    kind: { type: "string" },
                    producerProjectNumber: {
                        description: "The producer project number used to generate PIA tokens",
                        type: "string",
                    },
                    recaptchaSiteKey: {
                        description: "The reCAPTCHA v2 site key used to invoke the reCAPTCHA service. Always present.",
                        type: "string",
                    },
                    recaptchaStoken: { type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1GetSessionCookiePublicKeysResponse: {
                description: "Response message for GetSessionCookiePublicKeys.",
                properties: {
                    keys: {
                        description: "Public keys of the session cookie signer, formatted as [JSON Web Keys (JWK)](https://tools.ietf.org/html/rfc7517).",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1OpenIdConnectKey" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1IdpConfig: {
                description: "Config of an identity provider.",
                properties: {
                    clientId: { description: "OAuth2 client ID.", type: "string" },
                    enabled: {
                        description: "True if allows the user to sign in with the provider.",
                        type: "boolean",
                    },
                    experimentPercent: {
                        description: "Percent of users who will be prompted/redirected federated login for this IdP",
                        format: "int32",
                        type: "integer",
                    },
                    provider: {
                        description: "Name of the identity provider.",
                        enum: [
                            "PROVIDER_UNSPECIFIED",
                            "MSLIVE",
                            "GOOGLE",
                            "FACEBOOK",
                            "PAYPAL",
                            "TWITTER",
                            "YAHOO",
                            "AOL",
                            "GITHUB",
                            "GOOGLE_PLAY_GAMES",
                            "LINKEDIN",
                            "IOS_GAME_CENTER",
                        ],
                        type: "string",
                    },
                    secret: { description: "OAuth2 client secret.", type: "string" },
                    whitelistedAudiences: {
                        description: "Whitelisted client IDs for audience check.",
                        items: { type: "string" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1IssueSamlResponseRequest: {
                description: "Request message for IssueSamlResponse.",
                properties: {
                    idToken: {
                        description: "The Identity Platform ID token. It will be verified and then converted to a new SAMLResponse.",
                        type: "string",
                    },
                    rpId: {
                        description: "Relying Party identifier, which is the audience of issued SAMLResponse.",
                        type: "string",
                    },
                    samlAppEntityId: {
                        description: "SAML app entity id specified in Google Admin Console for each app. If developers want to redirect to a third-party app rather than a G Suite app, they'll probably they need this. When it's used, we'll return a RelayState. This includes a SAMLRequest, which can be used to trigger a SP-initiated SAML flow to redirect to the real app.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1IssueSamlResponseResponse: {
                description: "Response for IssueSamlResponse request.",
                properties: {
                    acsEndpoint: {
                        description: "The ACS endpoint which consumes the returned SAMLResponse.",
                        type: "string",
                    },
                    email: { description: "Email of the user.", type: "string" },
                    firstName: { description: "First name of the user.", type: "string" },
                    isNewUser: {
                        description: "Whether the logged in user was created by this request.",
                        type: "boolean",
                    },
                    lastName: { description: "Last name of the user.", type: "string" },
                    relayState: { description: "Generated RelayState.", type: "string" },
                    samlResponse: {
                        description: "Signed SAMLResponse created for the Relying Party.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1MfaEnrollment: {
                description: "Information on which multi-factor authentication (MFA) providers are enabled for an account.",
                properties: {
                    displayName: {
                        description: 'Display name for this mfa option e.g. "corp cell phone".',
                        type: "string",
                    },
                    enrolledAt: {
                        description: "Timestamp when the account enrolled this second factor.",
                        format: "google-datetime",
                        type: "string",
                    },
                    mfaEnrollmentId: { description: "ID of this MFA option.", type: "string" },
                    phoneInfo: {
                        description: "Normally this will show the phone number associated with this enrollment. In some situations, such as after a first factor sign in, it will only show the obfuscated version of the associated phone number.",
                        type: "string",
                    },
                    unobfuscatedPhoneInfo: {
                        description: "Output only. Unobfuscated phone_info.",
                        readOnly: true,
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1MfaFactor: {
                properties: {
                    displayName: {
                        description: 'Display name for this mfa option e.g. "corp cell phone".',
                        type: "string",
                    },
                    phoneInfo: { description: "Phone number to receive OTP for MFA.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1MfaInfo: {
                description: "Multi-factor authentication related information.",
                properties: {
                    enrollments: {
                        description: "The second factors the user has enrolled.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaEnrollment" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1OpenIdConnectKey: {
                description: "Represents a public key of the session cookie signer, formatted as a [JSON Web Key (JWK)](https://tools.ietf.org/html/rfc7517).",
                properties: {
                    alg: { description: "Signature algorithm.", type: "string" },
                    e: {
                        description: "Exponent for the RSA public key, it is represented as the base64url encoding of the value's big endian representation.",
                        type: "string",
                    },
                    kid: { description: "Unique string to identify this key.", type: "string" },
                    kty: { description: "Key type.", type: "string" },
                    n: {
                        description: "Modulus for the RSA public key, it is represented as the base64url encoding of the value's big endian representation.",
                        type: "string",
                    },
                    use: { description: "Key use.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1ProviderUserInfo: {
                description: "Information about the user as provided by various Identity Providers.",
                properties: {
                    displayName: {
                        description: "The user's display name at the Identity Provider.",
                        type: "string",
                    },
                    email: {
                        description: "The user's email address at the Identity Provider.",
                        type: "string",
                    },
                    federatedId: {
                        description: "The user's identifier at the Identity Provider.",
                        type: "string",
                    },
                    phoneNumber: {
                        description: "The user's phone number at the Identity Provider.",
                        type: "string",
                    },
                    photoUrl: {
                        description: "The user's profile photo URL at the Identity Provider.",
                        type: "string",
                    },
                    providerId: { description: "The ID of the Identity Provider.", type: "string" },
                    rawId: {
                        description: "The user's raw identifier directly returned from Identity Provider.",
                        type: "string",
                    },
                    screenName: {
                        description: "The user's screen_name at Twitter or login name at GitHub.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1QueryUserInfoRequest: {
                description: "Request message for QueryUserInfo.",
                properties: {
                    expression: {
                        description: "Query conditions used to filter results. If more than one is passed, only the first SqlExpression is evaluated.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SqlExpression" },
                        type: "array",
                    },
                    limit: {
                        description: "The maximum number of accounts to return with an upper limit of __500__. Defaults to _500_. Only valid when `return_user_info` is set to `true`.",
                        format: "int64",
                        type: "string",
                    },
                    offset: {
                        description: "The number of accounts to skip from the beginning of matching records. Only valid when `return_user_info` is set to `true`.",
                        format: "int64",
                        type: "string",
                    },
                    order: {
                        description: "The order for sorting query result. Defaults to __ascending__ order. Only valid when `return_user_info` is set to `true`.",
                        enum: ["ORDER_UNSPECIFIED", "ASC", "DESC"],
                        type: "string",
                    },
                    returnUserInfo: {
                        description: "If `true`, this request will return the accounts matching the query. If `false`, only the __count__ of accounts matching the query will be returned. Defaults to `true`.",
                        type: "boolean",
                    },
                    sortBy: {
                        description: "The field to use for sorting user accounts. Defaults to `USER_ID`. Note: when `phone_number` is specified in `expression`, the result ignores the sorting. Only valid when `return_user_info` is set to `true`.",
                        enum: [
                            "SORT_BY_FIELD_UNSPECIFIED",
                            "USER_ID",
                            "NAME",
                            "CREATED_AT",
                            "LAST_LOGIN_AT",
                            "USER_EMAIL",
                        ],
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the tenant to which the result is scoped.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1QueryUserInfoResponse: {
                description: "Response message for QueryUserInfo.",
                properties: {
                    recordsCount: {
                        description: "If `return_user_info` in the request is true, this is the number of returned accounts in this message. Otherwise, this is the total number of accounts matching the query.",
                        format: "int64",
                        type: "string",
                    },
                    userInfo: {
                        description: "If `return_user_info` in the request is true, this is the accounts matching the query.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1UserInfo" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1ResetPasswordRequest: {
                description: "Request message for ResetPassword.",
                properties: {
                    email: {
                        description: "The email of the account to be modified. Specify this and the old password in order to change an account's password without using an out-of-band code.",
                        type: "string",
                    },
                    newPassword: {
                        description: "The new password to be set for this account. Specifying this field will result in a change to the account and consume the out-of-band code if one was specified and it was of type PASSWORD_RESET.",
                        type: "string",
                    },
                    oldPassword: {
                        description: "The current password of the account to be modified. Specify this and email to change an account's password without using an out-of-band code.",
                        type: "string",
                    },
                    oobCode: {
                        description: "An out-of-band (OOB) code generated by GetOobCode request. Specify only this parameter (or only this parameter and a tenant ID) to get the out-of-band code's type in the response without mutating the account's state. Only a PASSWORD_RESET out-of-band code can be consumed via this method.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The tenant ID of the Identity Platform tenant the account belongs to.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1ResetPasswordResponse: {
                description: "Response message for ResetPassword.",
                properties: {
                    email: {
                        description: "The email associated with the out-of-band code that was used.",
                        type: "string",
                    },
                    kind: { type: "string" },
                    mfaInfo: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaEnrollment" },
                    newEmail: { type: "string" },
                    requestType: {
                        enum: [
                            "OOB_REQ_TYPE_UNSPECIFIED",
                            "PASSWORD_RESET",
                            "OLD_EMAIL_AGREE",
                            "NEW_EMAIL_ACCEPT",
                            "VERIFY_EMAIL",
                            "RECOVER_EMAIL",
                            "EMAIL_SIGNIN",
                            "VERIFY_AND_CHANGE_EMAIL",
                            "REVERT_SECOND_FACTOR_ADDITION",
                        ],
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SendVerificationCodeRequest: {
                description: "Request message for SendVerificationCode. At least one of (`ios_receipt` and `ios_secret`), `recaptcha_token`, or `safety_net_token` must be specified to verify the verification code is being sent on behalf of a real app and not an emulator.",
                properties: {
                    autoRetrievalInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1AutoRetrievalInfo",
                    },
                    iosReceipt: {
                        description: "Receipt of successful iOS app token validation. At least one of (`ios_receipt` and `ios_secret`), `recaptcha_token`, or `safety_net_token` must be specified to verify the verification code is being sent on behalf of a real app and not an emulator. This should come from the response of verifyIosClient. If present, the caller should also provide the `ios_secret`, as well as a bundle ID in the `x-ios-bundle-identifier` header, which must match the bundle ID from the verifyIosClient request.",
                        type: "string",
                    },
                    iosSecret: {
                        description: "Secret delivered to iOS app as a push notification. Should be passed with an `ios_receipt` as well as the `x-ios-bundle-identifier` header.",
                        type: "string",
                    },
                    phoneNumber: {
                        description: "The phone number to send the verification code to in E.164 format.",
                        type: "string",
                    },
                    recaptchaToken: {
                        description: "Recaptcha token for app verification. At least one of (`ios_receipt` and `ios_secret`), `recaptcha_token`, or `safety_net_token` must be specified to verify the verification code is being sent on behalf of a real app and not an emulator. The recaptcha should be generated by calling getRecaptchaParams and the recaptcha token will be generated on user completion of the recaptcha challenge.",
                        type: "string",
                    },
                    safetyNetToken: {
                        description: "Android only. Used to assert application identity in place of a recaptcha token. At least one of (`ios_receipt` and `ios_secret`), `recaptcha_token`, or `safety_net_token` must be specified to verify the verification code is being sent on behalf of a real app and not an emulator. A SafetyNet Token can be generated via the [SafetyNet Android Attestation API](https://developer.android.com/training/safetynet/attestation.html), with the Base64 encoding of the `phone_number` field as the nonce.",
                        type: "string",
                    },
                    tenantId: {
                        description: "Tenant ID of the Identity Platform tenant the user is signing in to.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SendVerificationCodeResponse: {
                description: "Response message for SendVerificationCode.",
                properties: {
                    sessionInfo: {
                        description: "Encrypted session information. This can be used in signInWithPhoneNumber to authenticate the phone number.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SetAccountInfoRequest: {
                description: "Request message for SetAccountInfo.",
                properties: {
                    captchaChallenge: { type: "string" },
                    captchaResponse: {
                        description: "The response from reCaptcha challenge. This is required when the system detects possible abuse activities.",
                        type: "string",
                    },
                    createdAt: {
                        description: "The timestamp in milliseconds when the account was created.",
                        format: "int64",
                        type: "string",
                    },
                    customAttributes: {
                        description: "JSON formatted custom attributes to be stored in the Identity Platform ID token. Specifying this field requires a Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "string",
                    },
                    delegatedProjectNumber: { format: "int64", type: "string" },
                    deleteAttribute: {
                        description: "The account's attributes to be deleted.",
                        items: {
                            enum: [
                                "USER_ATTRIBUTE_NAME_UNSPECIFIED",
                                "EMAIL",
                                "DISPLAY_NAME",
                                "PROVIDER",
                                "PHOTO_URL",
                                "PASSWORD",
                                "RAW_USER_INFO",
                            ],
                            type: "string",
                        },
                        type: "array",
                    },
                    deleteProvider: {
                        description: "The Identity Providers to unlink from the user's account.",
                        items: { type: "string" },
                        type: "array",
                    },
                    disableUser: {
                        description: "If true, marks the account as disabled, meaning the user will no longer be able to sign-in.",
                        type: "boolean",
                    },
                    displayName: {
                        description: "The user's new display name to be updated in the account's attributes. The length of the display name must be less than or equal to 256 characters.",
                        type: "string",
                    },
                    email: {
                        description: "The user's new email to be updated in the account's attributes. The length of email should be less than 256 characters and in the format of `name@domain.tld`. The email should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec production.",
                        type: "string",
                    },
                    emailVerified: {
                        description: "Whether the user's email has been verified. Specifying this field requires a Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "boolean",
                    },
                    idToken: {
                        description: "A valid Identity Platform ID token. Required when attempting to change user-related information.",
                        type: "string",
                    },
                    instanceId: { type: "string" },
                    lastLoginAt: {
                        description: "The timestamp in milliseconds when the account last logged in.",
                        format: "int64",
                        type: "string",
                    },
                    linkProviderUserInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1ProviderUserInfo",
                    },
                    localId: {
                        description: "The ID of the user. Specifying this field requires a Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control). For requests from end-users, an ID token should be passed instead.",
                        type: "string",
                    },
                    mfa: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaInfo" },
                    oobCode: {
                        description: "The out-of-band code to be applied on the user's account. The following out-of-band code types are supported: * VERIFY_EMAIL * RECOVER_EMAIL * REVERT_SECOND_FACTOR_ADDITION * VERIFY_AND_CHANGE_EMAIL",
                        type: "string",
                    },
                    password: {
                        description: "The user's new password to be updated in the account's attributes. The password must be at least 6 characters long.",
                        type: "string",
                    },
                    phoneNumber: {
                        description: "The phone number to be updated in the account's attributes.",
                        type: "string",
                    },
                    photoUrl: {
                        description: "The user's new photo URL for the account's profile photo to be updated in the account's attributes. The length of the URL must be less than or equal to 2048 characters.",
                        type: "string",
                    },
                    provider: {
                        description: "The Identity Providers that the account should be associated with.",
                        items: { type: "string" },
                        type: "array",
                    },
                    returnSecureToken: {
                        description: "Whether or not to return an ID and refresh token. Should always be true.",
                        type: "boolean",
                    },
                    targetProjectId: {
                        description: "The project ID for the project that the account belongs to. Specifying this field requires Google OAuth 2.0 credential with proper [permissions] (https://cloud.google.com/identity-platform/docs/access-control). Requests from end users should pass an Identity Platform ID token instead.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The tenant ID of the Identity Platform tenant that the account belongs to. Requests from end users should pass an Identity Platform ID token rather than setting this field.",
                        type: "string",
                    },
                    upgradeToFederatedLogin: {
                        description: "Whether the account should be restricted to only using federated login.",
                        type: "boolean",
                    },
                    validSince: {
                        description: "Specifies the minimum timestamp in seconds for an Identity Platform ID token to be considered valid.",
                        format: "int64",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SetAccountInfoResponse: {
                description: "Response message for SetAccountInfo",
                properties: {
                    displayName: { description: "The account's display name.", type: "string" },
                    email: { description: "The account's email address.", type: "string" },
                    emailVerified: {
                        description: "Whether the account's email has been verified.",
                        type: "boolean",
                    },
                    expiresIn: {
                        description: "The number of seconds until the Identity Platform ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    idToken: {
                        description: "An Identity Platform ID token for the account. This is used for legacy user sign up.",
                        type: "string",
                    },
                    kind: { type: "string" },
                    localId: { description: "The ID of the authenticated user.", type: "string" },
                    newEmail: {
                        description: "The new email that has been set on the user's account attributes.",
                        type: "string",
                    },
                    passwordHash: {
                        description: "Deprecated. No actual password hash is currently returned.",
                        type: "string",
                    },
                    photoUrl: {
                        description: "The user's photo URL for the account's profile photo.",
                        type: "string",
                    },
                    providerUserInfo: {
                        description: "The linked Identity Providers on the account.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1ProviderUserInfo" },
                        type: "array",
                    },
                    refreshToken: {
                        description: "A refresh token for the account. This is used for legacy user sign up.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithCustomTokenRequest: {
                description: "Request message for SignInWithCustomToken.",
                properties: {
                    delegatedProjectNumber: { format: "int64", type: "string" },
                    instanceId: { type: "string" },
                    returnSecureToken: { description: "Should always be true.", type: "boolean" },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to. If present, the ID should match the tenant_id in the token.",
                        type: "string",
                    },
                    token: {
                        description: 'Required. The custom Auth token asserted by the developer. The token should be a [JSON Web Token (JWT)](https://tools.ietf.org/html/rfc7519) that includes the claims listed in the [API reference](https://cloud.google.com/identity-platform/docs/reference/rest/client/) under the "Custom Token Claims" section.',
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithCustomTokenResponse: {
                description: "Response message for SignInWithCustomToken.",
                properties: {
                    expiresIn: {
                        description: "The number of seconds until the ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    idToken: {
                        description: "An Identity Platform ID token for the authenticated user.",
                        type: "string",
                    },
                    isNewUser: {
                        description: "Whether the authenticated user was created by this request.",
                        type: "boolean",
                    },
                    kind: { type: "string" },
                    refreshToken: {
                        description: "An Identity Platform refresh token for the authenticated user.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithEmailLinkRequest: {
                description: "Request message for SignInWithEmailLink",
                properties: {
                    email: {
                        description: "Required. The email address the sign-in link was sent to. The length of email should be less than 256 characters and in the format of `name@domain.tld`. The email should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec production.",
                        type: "string",
                    },
                    idToken: {
                        description: "A valid ID token for an Identity Platform account. If passed, this request will link the email address to the user represented by this ID token and enable sign-in with email link on the account for the future.",
                        type: "string",
                    },
                    oobCode: {
                        description: "Required. The out-of-band code from the email link.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to. If not set, the user will sign in to the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithEmailLinkResponse: {
                description: "Response message for SignInWithEmailLink.",
                properties: {
                    email: {
                        description: "The email the user signed in with. Always present in the response.",
                        type: "string",
                    },
                    expiresIn: {
                        description: "The number of seconds until the ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    idToken: {
                        description: "An Identity Platform ID token for the authenticated user.",
                        type: "string",
                    },
                    isNewUser: {
                        description: "Whether the authenticated user was created by this request.",
                        type: "boolean",
                    },
                    kind: { type: "string" },
                    localId: {
                        description: "The ID of the authenticated user. Always present in the response.",
                        type: "string",
                    },
                    mfaInfo: {
                        description: "Info on which multi-factor authentication providers are enabled. Present if the user needs to complete the sign-in using multi-factor authentication.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaEnrollment" },
                        type: "array",
                    },
                    mfaPendingCredential: {
                        description: "An opaque string that functions as proof that the user has successfully passed the first factor check.",
                        type: "string",
                    },
                    refreshToken: {
                        description: "Refresh token for the authenticated user.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithGameCenterRequest: {
                description: "Request message for SignInWithGameCenter",
                properties: {
                    displayName: { description: "The user's Game Center display name.", type: "string" },
                    gamePlayerId: {
                        description: "The user's Game Center game player ID. A unique identifier for a player of the game. https://developer.apple.com/documentation/gamekit/gkplayer/3113960-gameplayerid",
                        type: "string",
                    },
                    idToken: {
                        description: "A valid ID token for an Identity Platform account. If present, this request will link the Game Center player ID to the account represented by this ID token.",
                        type: "string",
                    },
                    playerId: { description: "Required. The user's Game Center player ID.", type: "string" },
                    publicKeyUrl: {
                        description: "Required. The URL to fetch the Apple public key in order to verify the given signature is signed by Apple.",
                        type: "string",
                    },
                    salt: {
                        description: "Required. A random string used to generate the given signature.",
                        type: "string",
                    },
                    signature: {
                        description: "Required. The verification signature data generated by Apple.",
                        type: "string",
                    },
                    teamPlayerId: {
                        description: "The user's Game Center team player ID. A unique identifier for a player of all the games that you distribute using your developer account. https://developer.apple.com/documentation/gamekit/gkplayer/3174857-teamplayerid",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to.",
                        type: "string",
                    },
                    timestamp: {
                        description: "Required. The time when the signature was created by Apple, in milliseconds since the epoch.",
                        format: "int64",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithGameCenterResponse: {
                description: "Response message for SignInWithGameCenter",
                properties: {
                    displayName: { description: "Display name of the authenticated user.", type: "string" },
                    expiresIn: {
                        description: "The number of seconds until the ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    gamePlayerId: {
                        description: "The user's Game Center game player ID. A unique identifier for a player of the game. https://developer.apple.com/documentation/gamekit/gkplayer/3113960-gameplayerid",
                        type: "string",
                    },
                    idToken: {
                        description: "An Identity Platform ID token for the authenticated user.",
                        type: "string",
                    },
                    isNewUser: {
                        description: "Whether the logged in user was created by this request.",
                        type: "boolean",
                    },
                    localId: {
                        description: "The ID of the authenticated user. Always present in the response.",
                        type: "string",
                    },
                    playerId: { description: "The user's Game Center player ID.", type: "string" },
                    refreshToken: {
                        description: "An Identity Platform refresh token for the authenticated user.",
                        type: "string",
                    },
                    teamPlayerId: {
                        description: "The user's Game Center team player ID. A unique identifier for a player of all the games that you distribute using your developer account. https://developer.apple.com/documentation/gamekit/gkplayer/3174857-teamplayerid",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithIdpRequest: {
                description: "Request message for SignInWithIdp.",
                properties: {
                    autoCreate: { type: "boolean" },
                    delegatedProjectNumber: { format: "int64", type: "string" },
                    idToken: {
                        description: "A valid Identity Platform ID token. If passed, the user's account at the IdP will be linked to the account represented by this ID token.",
                        type: "string",
                    },
                    pendingIdToken: { type: "string" },
                    pendingToken: {
                        description: "An opaque string from a previous SignInWithIdp response. If set, it can be used to repeat the sign-in operation from the previous SignInWithIdp operation.",
                        type: "string",
                    },
                    postBody: {
                        description: "If the user is signing in with an authorization response obtained via a previous CreateAuthUri authorization request, this is the body of the HTTP POST callback from the IdP, if present. Otherwise, if the user is signing in with a manually provided IdP credential, this should be a URL-encoded form that contains the credential (e.g. an ID token or access token for OAuth 2.0 IdPs) and the provider ID of the IdP that issued the credential. For example, if the user is signing in to the Google provider using a Google ID token, this should be set to `id_token=[GOOGLE_ID_TOKEN]&providerId=google.com`, where `[GOOGLE_ID_TOKEN]` should be replaced with the Google ID token. If the user is signing in to the Facebook provider using a Facebook authentication token, this should be set to `id_token=[FACEBOOK_AUTHENTICATION_TOKEN]&providerId=facebook.com&nonce= [NONCE]`, where `[FACEBOOK_AUTHENTICATION_TOKEN]` should be replaced with the Facebook authentication token. Nonce is required for validating the token. The request will fail if no nonce is provided. If the user is signing in to the Facebook provider using a Facebook access token, this should be set to `access_token=[FACEBOOK_ACCESS_TOKEN]&providerId=facebook.com`, where `[FACEBOOK_ACCESS_TOKEN]` should be replaced with the Facebook access token. If the user is signing in to the Twitter provider using a Twitter OAuth 1.0 credential, this should be set to `access_token=[TWITTER_ACCESS_TOKEN]&oauth_token_secret=[TWITTER_TOKEN_SECRET]&providerId=twitter.com`, where `[TWITTER_ACCESS_TOKEN]` and `[TWITTER_TOKEN_SECRET]` should be replaced with the Twitter OAuth access token and Twitter OAuth token secret respectively.",
                        type: "string",
                    },
                    requestUri: {
                        description: "Required. The URL to which the IdP redirects the user back. This can be set to `http://localhost` if the user is signing in with a manually provided IdP credential.",
                        type: "string",
                    },
                    returnIdpCredential: {
                        description: "Whether or not to return OAuth credentials from the IdP on the following errors: `FEDERATED_USER_ID_ALREADY_LINKED` and `EMAIL_EXISTS`.",
                        type: "boolean",
                    },
                    returnRefreshToken: {
                        description: "Whether or not to return the OAuth refresh token from the IdP, if available.",
                        type: "boolean",
                    },
                    returnSecureToken: { description: "Should always be true.", type: "boolean" },
                    sessionId: {
                        description: "The session ID returned from a previous CreateAuthUri call. This field is verified against that session ID to prevent session fixation attacks. Required if the user is signing in with an authorization response from a previous CreateAuthUri authorization request.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to. If not set, the user will sign in to the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithIdpResponse: {
                description: "Response message for SignInWithIdp.",
                properties: {
                    context: {
                        description: "The opaque string set in CreateAuthUri that is used to maintain contextual information between the authentication request and the callback from the IdP.",
                        type: "string",
                    },
                    dateOfBirth: {
                        description: "The date of birth for the user's account at the IdP.",
                        type: "string",
                    },
                    displayName: {
                        description: "The display name for the user's account at the IdP.",
                        type: "string",
                    },
                    email: {
                        description: "The email address of the user's account at the IdP.",
                        type: "string",
                    },
                    emailRecycled: {
                        description: 'Whether or not there is an existing Identity Platform user account with the same email address but linked to a different account at the same IdP. Only present if the "One account per email address" setting is enabled and the email address at the IdP is verified.',
                        type: "boolean",
                    },
                    emailVerified: {
                        description: "Whether the user account's email address is verified.",
                        type: "boolean",
                    },
                    errorMessage: {
                        description: "The error message returned if `return_idp_credential` is set to `true` and either the `FEDERATED_USER_ID_ALREADY_LINKED` or `EMAIL_EXISTS` error is encountered. This field's value is either `FEDERATED_USER_ID_ALREADY_LINKED` or `EMAIL_EXISTS`.",
                        type: "string",
                    },
                    expiresIn: {
                        description: "The number of seconds until the Identity Platform ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    federatedId: {
                        description: "The user's account ID at the IdP. Always present in the response.",
                        type: "string",
                    },
                    firstName: {
                        description: "The first name for the user's account at the IdP.",
                        type: "string",
                    },
                    fullName: {
                        description: "The full name for the user's account at the IdP.",
                        type: "string",
                    },
                    idToken: {
                        description: "An Identity Platform ID token for the authenticated user.",
                        type: "string",
                    },
                    inputEmail: { type: "string" },
                    isNewUser: {
                        description: "Whether or not a new Identity Platform account was created for the authenticated user.",
                        type: "boolean",
                    },
                    kind: { type: "string" },
                    language: {
                        description: "The language preference for the user's account at the IdP.",
                        type: "string",
                    },
                    lastName: {
                        description: "The last name for the user's account at the IdP.",
                        type: "string",
                    },
                    localId: {
                        description: "The ID of the authenticated Identity Platform user. Always present in the response.",
                        type: "string",
                    },
                    mfaInfo: {
                        description: "Info on which multi-factor authentication providers are enabled for the account. Present if the user needs to complete the sign-in using multi-factor authentication.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaEnrollment" },
                        type: "array",
                    },
                    mfaPendingCredential: {
                        description: "An opaque string that functions as proof that the user has successfully passed the first factor authentication.",
                        type: "string",
                    },
                    needConfirmation: {
                        description: 'Whether or not there is an existing Identity Platform user account with the same email address as the current account signed in at the IdP, and the account\'s email addresss is not verified at the IdP. The user will need to sign in to the existing Identity Platform account and then link the current credential from the IdP to it. Only present if the "One account per email address" setting is enabled.',
                        type: "boolean",
                    },
                    needEmail: { type: "boolean" },
                    nickName: {
                        description: "The nickname for the user's account at the IdP.",
                        type: "string",
                    },
                    oauthAccessToken: {
                        description: "The OAuth access token from the IdP, if available.",
                        type: "string",
                    },
                    oauthAuthorizationCode: {
                        description: "The OAuth 2.0 authorization code, if available. Only present for the Google provider.",
                        type: "string",
                    },
                    oauthExpireIn: {
                        description: "The number of seconds until the OAuth access token from the IdP expires.",
                        format: "int32",
                        type: "integer",
                    },
                    oauthIdToken: {
                        description: "The OpenID Connect ID token from the IdP, if available.",
                        type: "string",
                    },
                    oauthRefreshToken: {
                        description: "The OAuth 2.0 refresh token from the IdP, if available and `return_refresh_token` is set to `true`.",
                        type: "string",
                    },
                    oauthTokenSecret: {
                        description: "The OAuth 1.0 token secret from the IdP, if available. Only present for the Twitter provider.",
                        type: "string",
                    },
                    originalEmail: {
                        description: 'The main (top-level) email address of the user\'s Identity Platform account, if different from the email address at the IdP. Only present if the "One account per email address" setting is enabled.',
                        type: "string",
                    },
                    pendingToken: {
                        description: "An opaque string that can be used as a credential from the IdP the user is signing into. The pending token obtained here can be set in a future SignInWithIdp request to sign the same user in with the IdP again.",
                        type: "string",
                    },
                    photoUrl: {
                        description: "The URL of the user's profile picture at the IdP.",
                        type: "string",
                    },
                    providerId: {
                        description: "The provider ID of the IdP that the user is signing in to. Always present in the response.",
                        type: "string",
                    },
                    rawUserInfo: {
                        description: "The stringified JSON response containing all the data corresponding to the user's account at the IdP.",
                        type: "string",
                    },
                    refreshToken: {
                        description: "An Identity Platform refresh token for the authenticated user.",
                        type: "string",
                    },
                    screenName: {
                        description: "The screen name for the user's account at the Twitter IdP or the login name for the user's account at the GitHub IdP.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The value of the `tenant_id` field in the request.",
                        type: "string",
                    },
                    timeZone: {
                        description: "The time zone for the user's account at the IdP.",
                        type: "string",
                    },
                    verifiedProvider: {
                        description: "A list of provider IDs that the user can sign in to in order to resolve a `need_confirmation` error. Only present if `need_confirmation` is set to `true`.",
                        items: { type: "string" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithPasswordRequest: {
                description: "Request message for SignInWithPassword.",
                properties: {
                    captchaChallenge: { type: "string" },
                    captchaResponse: {
                        description: "The reCAPTCHA token provided by the reCAPTCHA client-side integration. reCAPTCHA Enterprise uses it for risk assessment. Required when reCAPTCHA Enterprise is enabled.",
                        type: "string",
                    },
                    delegatedProjectNumber: { format: "int64", type: "string" },
                    email: {
                        description: "Required. The email the user is signing in with. The length of email should be less than 256 characters and in the format of `name@domain.tld`. The email should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec production.",
                        type: "string",
                    },
                    idToken: { type: "string" },
                    instanceId: { type: "string" },
                    password: {
                        description: "Required. The password the user provides to sign in to the account.",
                        type: "string",
                    },
                    pendingIdToken: { type: "string" },
                    returnSecureToken: { description: "Should always be true.", type: "boolean" },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to. If not set, the user will sign in to the default Identity Platform instance in the project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithPasswordResponse: {
                description: "Response message for SignInWithPassword.",
                properties: {
                    displayName: {
                        description: "The user's display name stored in the account's attributes.",
                        type: "string",
                    },
                    email: {
                        description: "The email of the authenticated user. Always present in the response.",
                        type: "string",
                    },
                    expiresIn: {
                        description: "The number of seconds until the Identity Platform ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    idToken: {
                        description: "An Identity Platform ID token for the authenticated user.",
                        type: "string",
                    },
                    kind: { type: "string" },
                    localId: {
                        description: "The ID of the authenticated user. Always present in the response.",
                        type: "string",
                    },
                    mfaInfo: {
                        description: "Info on which multi-factor authentication providers are enabled for the account. Present if the user needs to complete the sign-in using multi-factor authentication.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaEnrollment" },
                        type: "array",
                    },
                    mfaPendingCredential: {
                        description: "An opaque string that functions as proof that the user has successfully passed the first factor authentication.",
                        type: "string",
                    },
                    oauthAccessToken: { description: "The OAuth2 access token.", type: "string" },
                    oauthAuthorizationCode: { type: "string" },
                    oauthExpireIn: {
                        description: "The access token expiration time in seconds.",
                        format: "int32",
                        type: "integer",
                    },
                    profilePicture: {
                        description: "The user's profile picture stored in the account's attributes.",
                        type: "string",
                    },
                    refreshToken: {
                        description: "An Identity Platform refresh token for the authenticated user.",
                        type: "string",
                    },
                    registered: {
                        description: "Whether the email is for an existing account. Always true.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithPhoneNumberRequest: {
                description: "Request message for SignInWithPhoneNumber.",
                properties: {
                    code: {
                        description: "User-entered verification code from an SMS sent to the user's phone.",
                        type: "string",
                    },
                    idToken: {
                        description: "A valid ID token for an Identity Platform account. If passed, this request will link the phone number to the user represented by this ID token if the phone number is not in use, or will reauthenticate the user if the phone number is already linked to the user.",
                        type: "string",
                    },
                    operation: {
                        enum: ["VERIFY_OP_UNSPECIFIED", "SIGN_UP_OR_IN", "REAUTH", "UPDATE", "LINK"],
                        type: "string",
                    },
                    phoneNumber: {
                        description: "The user's phone number to sign in with. This is necessary in the case of uing a temporary proof, in which case it must match the phone number that was authenticated in the request that generated the temporary proof. This field is ignored if a session info is passed.",
                        type: "string",
                    },
                    sessionInfo: {
                        description: "Encrypted session information from the response of sendVerificationCode. In the case of authenticating with an SMS code this must be specified, but in the case of using a temporary proof it can be unspecified.",
                        type: "string",
                    },
                    temporaryProof: {
                        description: "A proof of the phone number verification, provided from a previous signInWithPhoneNumber request. If this is passed, the caller must also pass in the phone_number field the phone number that was verified in the previous request.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to. If not set, the user will sign in to the default Identity Platform project.",
                        type: "string",
                    },
                    verificationProof: { description: "Do not use.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignInWithPhoneNumberResponse: {
                description: "Response message for SignInWithPhoneNumber.",
                properties: {
                    expiresIn: {
                        description: "The number of seconds until the ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    idToken: {
                        description: "Identity Platform ID token for the authenticated user.",
                        type: "string",
                    },
                    isNewUser: {
                        description: "Whether the authenticated user was created by this request.",
                        type: "boolean",
                    },
                    localId: {
                        description: "The id of the authenticated user. Present in the case of a successful authentication. In the case when the phone could be verified but the account operation could not be performed, a temporary proof will be returned instead.",
                        type: "string",
                    },
                    phoneNumber: {
                        description: "Phone number of the authenticated user. Always present in the response.",
                        type: "string",
                    },
                    refreshToken: {
                        description: "Refresh token for the authenticated user.",
                        type: "string",
                    },
                    temporaryProof: {
                        description: "A proof of the phone number verification, provided if a phone authentication is successful but the user operation fails. This happens when the request tries to link a phone number to a user with an ID token or reauthenticate with an ID token but the phone number is linked to a different user.",
                        type: "string",
                    },
                    temporaryProofExpiresIn: {
                        description: "The number of seconds until the temporary proof expires.",
                        format: "int64",
                        type: "string",
                    },
                    verificationProof: { description: "Do not use.", type: "string" },
                    verificationProofExpiresIn: {
                        description: "Do not use.",
                        format: "int64",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignUpRequest: {
                description: "Request message for SignUp.",
                properties: {
                    captchaChallenge: { type: "string" },
                    captchaResponse: {
                        description: "The reCAPTCHA token provided by the reCAPTCHA client-side integration. reCAPTCHA Enterprise uses it for assessment. Required when reCAPTCHA enterprise is enabled.",
                        type: "string",
                    },
                    disabled: {
                        description: "Whether the user will be disabled upon creation. Disabled accounts are inaccessible except for requests bearing a Google OAuth2 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "boolean",
                    },
                    displayName: {
                        description: "The display name of the user to be created.",
                        type: "string",
                    },
                    email: {
                        description: "The email to assign to the created user. The length of the email should be less than 256 characters and in the format of `name@domain.tld`. The email should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec production. An anonymous user will be created if not provided.",
                        type: "string",
                    },
                    emailVerified: {
                        description: "Whether the user's email is verified. Specifying this field requires a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "boolean",
                    },
                    idToken: {
                        description: "A valid ID token for an Identity Platform user. If set, this request will link the authentication credential to the user represented by this ID token. For a non-admin request, both the `email` and `password` fields must be set. For an admin request, `local_id` must not be set.",
                        type: "string",
                    },
                    instanceId: { type: "string" },
                    localId: {
                        description: "The ID of the user to create. The ID must be unique within the project that the user is being created under. Specifying this field requires a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "string",
                    },
                    mfaInfo: {
                        description: "The multi-factor authentication providers for the user to create.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaFactor" },
                        type: "array",
                    },
                    password: {
                        description: "The password to assign to the created user. The password must be be at least 6 characters long. If set, the `email` field must also be set.",
                        type: "string",
                    },
                    phoneNumber: {
                        description: "The phone number of the user to create. Specifying this field requires a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        type: "string",
                    },
                    photoUrl: { description: "The profile photo url of the user to create.", type: "string" },
                    targetProjectId: {
                        description: "The project ID of the project which the user should belong to. Specifying this field requires a Google OAuth 2.0 credential with the proper [permissions](https://cloud.google.com/identity-platform/docs/access-control). If this is not set, the target project is inferred from the scope associated to the Bearer access token.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant to create a user under. If not set, the user will be created under the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SignUpResponse: {
                description: "Response message for SignUp.",
                properties: {
                    displayName: { description: "The created user's display name.", type: "string" },
                    email: { description: "The created user's email.", type: "string" },
                    expiresIn: {
                        description: "The number of seconds until the ID token expires.",
                        format: "int64",
                        type: "string",
                    },
                    idToken: {
                        description: "An Identity Platform ID token for the created user. This field is only set for non-admin requests.",
                        type: "string",
                    },
                    kind: { type: "string" },
                    localId: {
                        description: "The ID of the created user. Always present in the response.",
                        type: "string",
                    },
                    refreshToken: {
                        description: "An Identity Platform refresh token for the created user. This field is only set for non-admin requests.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1SqlExpression: {
                description: "Query conditions used to filter results.",
                properties: {
                    email: {
                        description: "A case insensitive string that the account's email should match. Only one of `email`, `phone_number`, or `user_id` should be specified in a SqlExpression. If more than one is specified, only the first (in that order) will be applied.",
                        type: "string",
                    },
                    phoneNumber: {
                        description: "A string that the account's phone number should match. Only one of `email`, `phone_number`, or `user_id` should be specified in a SqlExpression. If more than one is specified, only the first (in that order) will be applied.",
                        type: "string",
                    },
                    userId: {
                        description: "A string that the account's local ID should match. Only one of `email`, `phone_number`, or `user_id` should be specified in a SqlExpression If more than one is specified, only the first (in that order) will be applied.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1UploadAccountRequest: {
                description: "Request message for UploadAccount.",
                properties: {
                    allowOverwrite: {
                        description: "Whether to overwrite an existing account in Identity Platform with a matching `local_id` in the request. If true, the existing account will be overwritten. If false, an error will be returned.",
                        type: "boolean",
                    },
                    argon2Parameters: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1Argon2Parameters",
                    },
                    blockSize: {
                        description: "The block size parameter used by the STANDARD_SCRYPT hashing function. This parameter, along with parallelization and cpu_mem_cost help tune the resources needed to hash a password, and should be tuned as processor speeds and memory technologies advance.",
                        format: "int32",
                        type: "integer",
                    },
                    cpuMemCost: {
                        description: "The CPU memory cost parameter to be used by the STANDARD_SCRYPT hashing function. This parameter, along with block_size and cpu_mem_cost help tune the resources needed to hash a password, and should be tuned as processor speeds and memory technologies advance.",
                        format: "int32",
                        type: "integer",
                    },
                    delegatedProjectNumber: { format: "int64", type: "string" },
                    dkLen: {
                        description: "The desired key length for the STANDARD_SCRYPT hashing function. Must be at least 1.",
                        format: "int32",
                        type: "integer",
                    },
                    hashAlgorithm: {
                        description: "Required. The hashing function used to hash the account passwords. Must be one of the following: * HMAC_SHA256 * HMAC_SHA1 * HMAC_MD5 * SCRYPT * PBKDF_SHA1 * MD5 * HMAC_SHA512 * SHA1 * BCRYPT * PBKDF2_SHA256 * SHA256 * SHA512 * STANDARD_SCRYPT * ARGON2",
                        type: "string",
                    },
                    memoryCost: {
                        description: "Memory cost for hash calculation. Only required when the hashing function is SCRYPT.",
                        format: "int32",
                        type: "integer",
                    },
                    parallelization: {
                        description: "The parallelization cost parameter to be used by the STANDARD_SCRYPT hashing function. This parameter, along with block_size and cpu_mem_cost help tune the resources needed to hash a password, and should be tuned as processor speeds and memory technologies advance.",
                        format: "int32",
                        type: "integer",
                    },
                    passwordHashOrder: {
                        description: "Password and salt order when verify password.",
                        enum: ["UNSPECIFIED_ORDER", "SALT_AND_PASSWORD", "PASSWORD_AND_SALT"],
                        type: "string",
                    },
                    rounds: {
                        description: "The number of rounds used for hash calculation. Only required for the following hashing functions: * MD5 * SHA1 * SHA256 * SHA512 * PBKDF_SHA1 * PBKDF2_SHA256 * SCRYPT",
                        format: "int32",
                        type: "integer",
                    },
                    saltSeparator: {
                        description: "One or more bytes to be inserted between the salt and plain text password. For stronger security, this should be a single non-printable character.",
                        format: "byte",
                        type: "string",
                    },
                    sanityCheck: {
                        description: "If true, the service will do the following list of checks before an account is uploaded: * Duplicate emails * Duplicate federated IDs * Federated ID provider validation If the duplication exists within the list of accounts to be uploaded, it will prevent the entire list from being uploaded. If the email or federated ID is a duplicate of a user already within the project/tenant, the account will not be uploaded, but the rest of the accounts will be unaffected. If false, these checks will be skipped.",
                        type: "boolean",
                    },
                    signerKey: {
                        description: "The signer key used to hash the password. Required for the following hashing functions: * SCRYPT, * HMAC_MD5, * HMAC_SHA1, * HMAC_SHA256, * HMAC_SHA512",
                        format: "byte",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the account belongs to.",
                        type: "string",
                    },
                    users: {
                        description: "A list of accounts to upload. `local_id` is required for each user; everything else is optional.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1UserInfo" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1UploadAccountResponse: {
                description: "Response message for UploadAccount.",
                properties: {
                    error: {
                        description: "Detailed error info for accounts that cannot be uploaded.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1ErrorInfo" },
                        type: "array",
                    },
                    kind: { type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1UserInfo: {
                description: "An Identity Platform account's information.",
                properties: {
                    createdAt: {
                        description: "The time, in milliseconds from epoch, when the account was created.",
                        format: "int64",
                        type: "string",
                    },
                    customAttributes: {
                        description: "Custom claims to be added to any ID tokens minted for the account. Should be at most 1,000 characters in length and in valid JSON format.",
                        type: "string",
                    },
                    customAuth: {
                        description: "Output only. Whether this account has been authenticated using SignInWithCustomToken.",
                        readOnly: true,
                        type: "boolean",
                    },
                    dateOfBirth: {
                        description: "Output only. The date of birth set for the account. This account attribute is not used by Identity Platform. It is available for informational purposes only.",
                        readOnly: true,
                        type: "string",
                    },
                    disabled: {
                        description: "Whether the account is disabled. Disabled accounts are inaccessible except for requests bearing a Google OAuth2 credential with proper permissions.",
                        type: "boolean",
                    },
                    displayName: {
                        description: "The display name of the account. This account attribute is not used by Identity Platform. It is available for informational purposes only.",
                        type: "string",
                    },
                    email: {
                        description: "The account's email address. The length of the email should be less than 256 characters and in the format of `name@domain.tld`. The email should also match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec.",
                        type: "string",
                    },
                    emailLinkSignin: {
                        description: "Output only. Whether the account can authenticate with email link.",
                        readOnly: true,
                        type: "boolean",
                    },
                    emailVerified: {
                        description: "Whether the account's email address has been verified.",
                        type: "boolean",
                    },
                    initialEmail: {
                        description: "The first email address associated with this account. The account's initial email cannot be changed once set and is used to recover access to this account if lost via the RECOVER_EMAIL flow in GetOobCode. Should match the [RFC 822](https://tools.ietf.org/html/rfc822) addr-spec.",
                        type: "string",
                    },
                    language: {
                        description: "Output only. The language preference of the account. This account attribute is not used by Identity Platform. It is available for informational purposes only.",
                        readOnly: true,
                        type: "string",
                    },
                    lastLoginAt: {
                        description: "The last time, in milliseconds from epoch, this account was logged into.",
                        format: "int64",
                        type: "string",
                    },
                    lastRefreshAt: {
                        description: "Timestamp when an ID token was last minted for this account.",
                        format: "google-datetime",
                        type: "string",
                    },
                    localId: { description: "Immutable. The unique ID of the account.", type: "string" },
                    mfaInfo: {
                        description: "Information on which multi-factor authentication providers are enabled for this account.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1MfaEnrollment" },
                        type: "array",
                    },
                    passwordHash: {
                        description: "The account's hashed password. Only accessible by requests bearing a Google OAuth2 credential with proper [permissions](https://cloud.google.com/identity-platform/docs/access-control).",
                        format: "byte",
                        type: "string",
                    },
                    passwordUpdatedAt: {
                        description: "The timestamp, in milliseconds from the epoch of 1970-01-01T00:00:00Z, when the account's password was last updated.",
                        format: "double",
                        type: "number",
                    },
                    phoneNumber: { description: "The account's phone number.", type: "string" },
                    photoUrl: {
                        description: "The URL of the account's profile photo. This account attribute is not used by Identity Platform. It is available for informational purposes only.",
                        type: "string",
                    },
                    providerUserInfo: {
                        description: "Information about the user as provided by various Identity Providers.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1ProviderUserInfo" },
                        type: "array",
                    },
                    rawPassword: {
                        description: "Input only. Plain text password used to update a account's password. This field is only ever used as input in a request. Identity Platform uses cryptographically secure hashing when managing passwords and will never store or transmit a user's password in plain text.",
                        type: "string",
                    },
                    salt: {
                        description: "The account's password salt. Only accessible by requests bearing a Google OAuth2 credential with proper permissions.",
                        format: "byte",
                        type: "string",
                    },
                    screenName: {
                        description: "Output only. This account's screen name at Twitter or login name at GitHub.",
                        readOnly: true,
                        type: "string",
                    },
                    tenantId: {
                        description: "ID of the tenant this account belongs to. Only set if this account belongs to a tenant.",
                        type: "string",
                    },
                    timeZone: {
                        description: "Output only. The time zone preference of the account. This account attribute is not used by Identity Platform. It is available for informational purposes only.",
                        readOnly: true,
                        type: "string",
                    },
                    validSince: {
                        description: "Oldest timestamp, in seconds since epoch, that an ID token should be considered valid. All ID tokens issued before this time are considered invalid.",
                        format: "int64",
                        type: "string",
                    },
                    version: {
                        description: "The version of the account's password. Only accessible by requests bearing a Google OAuth2 credential with proper permissions.",
                        format: "int32",
                        type: "integer",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1VerifyIosClientRequest: {
                description: "Request message for VerifyIosClient",
                properties: {
                    appToken: {
                        description: "A device token that the iOS client gets after registering to APNs (Apple Push Notification service).",
                        type: "string",
                    },
                    isSandbox: {
                        description: "Whether the app token is in the iOS sandbox. If false, the app token is in the production environment.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV1VerifyIosClientResponse: {
                description: "Response message for VerifyIosClient.",
                properties: {
                    receipt: { description: "Receipt of successful app token validation.", type: "string" },
                    suggestedTimeout: {
                        description: "Suggested time that the client should wait in seconds for delivery of the push notification.",
                        format: "int64",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2AllowByDefault: {
                description: "Defines a policy of allowing every region by default and adding disallowed regions to a disallow list.",
                properties: {
                    disallowedRegions: {
                        description: "Two letter unicode region codes to disallow as defined by https://cldr.unicode.org/ The full list of these region codes is here: https://github.com/unicode-cldr/cldr-localenames-full/blob/master/main/en/territories.json",
                        items: { type: "string" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2AllowlistOnly: {
                description: "Defines a policy of only allowing regions by explicitly adding them to an allowlist.",
                properties: {
                    allowedRegions: {
                        description: "Two letter unicode region codes to allow as defined by https://cldr.unicode.org/ The full list of these region codes is here: https://github.com/unicode-cldr/cldr-localenames-full/blob/master/main/en/territories.json",
                        items: { type: "string" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Anonymous: {
                description: "Configuration options related to authenticating an anonymous user.",
                properties: {
                    enabled: {
                        description: "Whether anonymous user auth is enabled for the project or not.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2AppleSignInConfig: {
                description: "Additional config for SignInWithApple.",
                properties: {
                    bundleIds: {
                        description: "A list of Bundle ID's usable by this project",
                        items: { type: "string" },
                        type: "array",
                    },
                    codeFlowConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2CodeFlowConfig",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2BlockingFunctionsConfig: {
                description: "Configuration related to Blocking Functions.",
                properties: {
                    forwardInboundCredentials: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ForwardInboundCredentials",
                    },
                    triggers: {
                        additionalProperties: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Trigger",
                        },
                        description: 'Map of Trigger to event type. Key should be one of the supported event types: "beforeCreate", "beforeSignIn"',
                        type: "object",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ClientConfig: {
                description: "Options related to how clients making requests on behalf of a project should be configured.",
                properties: {
                    apiKey: {
                        description: "Output only. API key that can be used when making requests for this project.",
                        readOnly: true,
                        type: "string",
                    },
                    firebaseSubdomain: {
                        description: "Output only. Firebase subdomain.",
                        readOnly: true,
                        type: "string",
                    },
                    permissions: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Permissions",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ClientPermissionConfig: {
                description: "Options related to how clients making requests on behalf of a tenant should be configured.",
                properties: {
                    permissions: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ClientPermissions",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ClientPermissions: {
                description: "Configuration related to restricting a user's ability to affect their account.",
                properties: {
                    disabledUserDeletion: {
                        description: "When true, end users cannot delete their account on the associated project through any of our API methods",
                        type: "boolean",
                    },
                    disabledUserSignup: {
                        description: "When true, end users cannot sign up for a new account on the associated project through any of our API methods",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2CodeFlowConfig: {
                description: "Additional config for Apple for code flow.",
                properties: {
                    keyId: { description: "Key ID for the private key.", type: "string" },
                    privateKey: {
                        description: "Private key used for signing the client secret JWT.",
                        type: "string",
                    },
                    teamId: { description: "Apple Developer Team ID.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Config: {
                description: "Represents an Identity Toolkit project.",
                properties: {
                    authorizedDomains: {
                        description: "List of domains authorized for OAuth redirects",
                        items: { type: "string" },
                        type: "array",
                    },
                    autodeleteAnonymousUsers: {
                        description: "Whether anonymous users will be auto-deleted after a period of 30 days.",
                        type: "boolean",
                    },
                    blockingFunctions: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2BlockingFunctionsConfig",
                    },
                    client: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ClientConfig" },
                    emailPrivacyConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2EmailPrivacyConfig",
                    },
                    mfa: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2MultiFactorAuthConfig",
                    },
                    monitoring: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2MonitoringConfig",
                    },
                    multiTenant: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2MultiTenantConfig",
                    },
                    name: {
                        description: 'Output only. The name of the Config resource. Example: "projects/my-awesome-project/config"',
                        readOnly: true,
                        type: "string",
                    },
                    notification: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2NotificationConfig",
                    },
                    quota: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2QuotaConfig" },
                    signIn: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SignInConfig" },
                    smsRegionConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SmsRegionConfig",
                    },
                    subtype: {
                        description: "Output only. The subtype of this config.",
                        enum: ["SUBTYPE_UNSPECIFIED", "IDENTITY_PLATFORM", "FIREBASE_AUTH"],
                        readOnly: true,
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdp: {
                description: "Standard Identity Toolkit-trusted IDPs.",
                properties: {
                    description: { description: "Description of the Idp", type: "string" },
                    idpId: { description: "Id the of Idp", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig: {
                description: "Configurations options for authenticating with a the standard set of Identity Toolkit-trusted IDPs.",
                properties: {
                    appleSignInConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2AppleSignInConfig",
                    },
                    clientId: { description: "OAuth client ID.", type: "string" },
                    clientSecret: { description: "OAuth client secret.", type: "string" },
                    enabled: {
                        description: "True if allows the user to sign in with the provider.",
                        type: "boolean",
                    },
                    name: {
                        description: 'The name of the DefaultSupportedIdpConfig resource, for example: "projects/my-awesome-project/defaultSupportedIdpConfigs/google.com"',
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2DnsInfo: {
                description: "Information of custom domain DNS verification. By default, default_domain will be used. A custom domain can be configured using VerifyCustomDomain.",
                properties: {
                    customDomain: {
                        description: "Output only. The applied verified custom domain.",
                        readOnly: true,
                        type: "string",
                    },
                    customDomainState: {
                        description: "Output only. The current verification state of the custom domain. The custom domain will only be used once the domain verification is successful.",
                        enum: [
                            "VERIFICATION_STATE_UNSPECIFIED",
                            "NOT_STARTED",
                            "IN_PROGRESS",
                            "FAILED",
                            "SUCCEEDED",
                        ],
                        readOnly: true,
                        type: "string",
                    },
                    domainVerificationRequestTime: {
                        description: "Output only. The timestamp of initial request for the current domain verification.",
                        format: "google-datetime",
                        readOnly: true,
                        type: "string",
                    },
                    pendingCustomDomain: {
                        description: "Output only. The custom domain that's to be verified.",
                        readOnly: true,
                        type: "string",
                    },
                    useCustomDomain: { description: "Whether to use custom domain.", type: "boolean" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Email: {
                description: "Configuration options related to authenticating a user by their email address.",
                properties: {
                    enabled: {
                        description: "Whether email auth is enabled for the project or not.",
                        type: "boolean",
                    },
                    passwordRequired: {
                        description: "Whether a password is required for email auth or not. If true, both an email and password must be provided to sign in. If false, a user may sign in via either email/password or email link.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2EmailPrivacyConfig: {
                description: "Configuration for settings related to email privacy and public visibility. Settings in this config protect against email enumeration, but may make some trade-offs in user-friendliness.",
                properties: {
                    enableImprovedEmailPrivacy: {
                        description: "Migrates the project to a state of improved email privacy. For example certain error codes are more generic to avoid giving away information on whether the account exists. In addition, this disables certain features that as a side-effect allow user enumeration. Enabling this toggle disables the fetchSignInMethodsForEmail functionality and changing the user's email to an unverified email. It is recommended to remove dependence on this functionality and enable this toggle to improve user privacy.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2EmailTemplate: {
                description: "Email template. The subject and body fields can contain the following placeholders which will be replaced with the appropriate values: %LINK% - The link to use to redeem the send OOB code. %EMAIL% - The email where the email is being sent. %NEW_EMAIL% - The new email being set for the account (when applicable). %APP_NAME% - The GCP project's display name. %DISPLAY_NAME% - The user's display name.",
                properties: {
                    body: { description: "Email body", type: "string" },
                    bodyFormat: {
                        description: "Email body format",
                        enum: ["BODY_FORMAT_UNSPECIFIED", "PLAIN_TEXT", "HTML"],
                        type: "string",
                    },
                    customized: {
                        description: "Output only. Whether the body or subject of the email is customized.",
                        readOnly: true,
                        type: "boolean",
                    },
                    replyTo: { description: "Reply-to address", type: "string" },
                    senderDisplayName: { description: "Sender display name", type: "string" },
                    senderLocalPart: { description: "Local part of From address", type: "string" },
                    subject: { description: "Subject of the email", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ForwardInboundCredentials: {
                description: "Indicates which credentials to pass to the registered Blocking Functions.",
                properties: {
                    accessToken: {
                        description: "Whether to pass the user's OAuth identity provider's access token.",
                        type: "boolean",
                    },
                    idToken: {
                        description: "Whether to pass the user's OIDC identity provider's ID token.",
                        type: "boolean",
                    },
                    refreshToken: {
                        description: "Whether to pass the user's OAuth identity provider's refresh token.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2HashConfig: {
                description: "History information of the hash algorithm and key. Different accounts' passwords may be generated by different version.",
                properties: {
                    algorithm: {
                        description: "Output only. Different password hash algorithms used in Identity Toolkit.",
                        enum: [
                            "HASH_ALGORITHM_UNSPECIFIED",
                            "HMAC_SHA256",
                            "HMAC_SHA1",
                            "HMAC_MD5",
                            "SCRYPT",
                            "PBKDF_SHA1",
                            "MD5",
                            "HMAC_SHA512",
                            "SHA1",
                            "BCRYPT",
                            "PBKDF2_SHA256",
                            "SHA256",
                            "SHA512",
                            "STANDARD_SCRYPT",
                        ],
                        readOnly: true,
                        type: "string",
                    },
                    memoryCost: {
                        description: "Output only. Memory cost for hash calculation. Used by scrypt and other similar password derivation algorithms. See https://tools.ietf.org/html/rfc7914 for explanation of field.",
                        format: "int32",
                        readOnly: true,
                        type: "integer",
                    },
                    rounds: {
                        description: "Output only. How many rounds for hash calculation. Used by scrypt and other similar password derivation algorithms.",
                        format: "int32",
                        readOnly: true,
                        type: "integer",
                    },
                    saltSeparator: {
                        description: "Output only. Non-printable character to be inserted between the salt and plain text password in base64.",
                        readOnly: true,
                        type: "string",
                    },
                    signerKey: {
                        description: "Output only. Signer key in base64.",
                        readOnly: true,
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2IdpCertificate: {
                description: "The IDP's certificate data to verify the signature in the SAMLResponse issued by the IDP.",
                properties: { x509Certificate: { description: "The x509 certificate", type: "string" } },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2IdpConfig: {
                description: "The SAML IdP (Identity Provider) configuration when the project acts as the relying party.",
                properties: {
                    idpCertificates: {
                        description: "IDP's public keys for verifying signature in the assertions.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2IdpCertificate" },
                        type: "array",
                    },
                    idpEntityId: { description: "Unique identifier for all SAML entities.", type: "string" },
                    signRequest: {
                        description: "Indicates if outbounding SAMLRequest should be signed.",
                        type: "boolean",
                    },
                    ssoUrl: { description: "URL to send Authentication request to.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig: {
                description: "A pair of SAML RP-IDP configurations when the project acts as the relying party.",
                properties: {
                    displayName: {
                        description: "The config's display name set by developers.",
                        type: "string",
                    },
                    enabled: {
                        description: "True if allows the user to sign in with the provider.",
                        type: "boolean",
                    },
                    idpConfig: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2IdpConfig" },
                    name: {
                        description: "The name of the InboundSamlConfig resource, for example: 'projects/my-awesome-project/inboundSamlConfigs/my-config-id'. Ignored during create requests.",
                        type: "string",
                    },
                    spConfig: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SpConfig" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Inheritance: {
                description: "Settings that the tenants will inherit from project level.",
                properties: {
                    emailSendingConfig: {
                        description: "Whether to allow the tenant to inherit custom domains, email templates, and custom SMTP settings. If true, email sent from tenant will follow the project level email sending configurations. If false (by default), emails will go with the default settings with no customizations.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2InitializeIdentityPlatformRequest: {
                description: "Request for InitializeIdentityPlatform.",
                properties: {},
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2InitializeIdentityPlatformResponse: {
                description: "Response for InitializeIdentityPlatform. Empty for now.",
                properties: {},
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ListDefaultSupportedIdpConfigsResponse: {
                description: "Response for DefaultSupportedIdpConfigs",
                properties: {
                    defaultSupportedIdpConfigs: {
                        description: "The set of configs.",
                        items: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                        },
                        type: "array",
                    },
                    nextPageToken: {
                        description: "Token to retrieve the next page of results, or empty if there are no more results in the list.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ListDefaultSupportedIdpsResponse: {
                description: "Response for ListDefaultSupportedIdps",
                properties: {
                    defaultSupportedIdps: {
                        description: "The set of configs.",
                        items: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdp",
                        },
                        type: "array",
                    },
                    nextPageToken: {
                        description: "Token to retrieve the next page of results, or empty if there are no more results in the list.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ListInboundSamlConfigsResponse: {
                description: "Response for ListInboundSamlConfigs",
                properties: {
                    inboundSamlConfigs: {
                        description: "The set of configs.",
                        items: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                        },
                        type: "array",
                    },
                    nextPageToken: {
                        description: "Token to retrieve the next page of results, or empty if there are no more results in the list.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ListOAuthIdpConfigsResponse: {
                description: "Response for ListOAuthIdpConfigs",
                properties: {
                    nextPageToken: {
                        description: "Token to retrieve the next page of results, or empty if there are no more results in the list.",
                        type: "string",
                    },
                    oauthIdpConfigs: {
                        description: "The set of configs.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2ListTenantsResponse: {
                description: "Response message for ListTenants.",
                properties: {
                    nextPageToken: {
                        description: "The token to get the next page of results.",
                        type: "string",
                    },
                    tenants: {
                        description: "A list of tenants under the given agent project.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Tenant" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2MonitoringConfig: {
                description: "Configuration related to monitoring project activity.",
                properties: {
                    requestLogging: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2RequestLogging",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2MultiFactorAuthConfig: {
                description: "Options related to MultiFactor Authentication for the project.",
                properties: {
                    enabledProviders: {
                        description: "A list of usable second factors for this project.",
                        items: { enum: ["PROVIDER_UNSPECIFIED", "PHONE_SMS"], type: "string" },
                        type: "array",
                    },
                    state: {
                        description: "Whether MultiFactor Authentication has been enabled for this project.",
                        enum: ["STATE_UNSPECIFIED", "DISABLED", "ENABLED", "MANDATORY"],
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2MultiTenantConfig: {
                description: "Configuration related to multi-tenant functionality.",
                properties: {
                    allowTenants: {
                        description: "Whether this project can have tenants or not.",
                        type: "boolean",
                    },
                    defaultTenantLocation: {
                        description: 'The default cloud parent org or folder that the tenant project should be created under. The parent resource name should be in the format of "/", such as "folders/123" or "organizations/456". If the value is not set, the tenant will be created under the same organization or folder as the agent project.',
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2NotificationConfig: {
                description: "Configuration related to sending notifications to users.",
                properties: {
                    defaultLocale: {
                        description: "Default locale used for email and SMS in IETF BCP 47 format.",
                        type: "string",
                    },
                    sendEmail: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SendEmail" },
                    sendSms: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SendSms" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig: {
                description: "Configuration options for authenticating with an OAuth IDP.",
                properties: {
                    clientId: { description: "The client id of an OAuth client.", type: "string" },
                    clientSecret: {
                        description: "The client secret of the OAuth client, to enable OIDC code flow.",
                        type: "string",
                    },
                    displayName: {
                        description: "The config's display name set by developers.",
                        type: "string",
                    },
                    enabled: {
                        description: "True if allows the user to sign in with the provider.",
                        type: "boolean",
                    },
                    issuer: { description: "For OIDC Idps, the issuer identifier.", type: "string" },
                    name: {
                        description: "The name of the OAuthIdpConfig resource, for example: 'projects/my-awesome-project/oauthIdpConfigs/oauth-config-id'. Ignored during create requests.",
                        type: "string",
                    },
                    responseType: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthResponseType",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2OAuthResponseType: {
                description: "The response type to request for in the OAuth authorization flow. You can set either `id_token` or `code` to true, but not both. Setting both types to be simultaneously true (`{code: true, id_token: true}`) is not yet supported. See https://openid.net/specs/openid-connect-core-1_0.html#Authentication for a mapping of response type to OAuth 2.0 flow.",
                properties: {
                    code: {
                        description: "If true, authorization code is returned from IdP's authorization endpoint.",
                        type: "boolean",
                    },
                    idToken: {
                        description: "If true, ID token is returned from IdP's authorization endpoint.",
                        type: "boolean",
                    },
                    token: {
                        description: "Do not use. The `token` response type is not supported at the moment.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Permissions: {
                description: "Configuration related to restricting a user's ability to affect their account.",
                properties: {
                    disabledUserDeletion: {
                        description: "When true, end users cannot delete their account on the associated project through any of our API methods",
                        type: "boolean",
                    },
                    disabledUserSignup: {
                        description: "When true, end users cannot sign up for a new account on the associated project through any of our API methods",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2PhoneNumber: {
                description: "Configuration options related to authenticated a user by their phone number.",
                properties: {
                    enabled: {
                        description: "Whether phone number auth is enabled for the project or not.",
                        type: "boolean",
                    },
                    testPhoneNumbers: {
                        additionalProperties: { type: "string" },
                        description: "A map of that can be used for phone auth testing.",
                        type: "object",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2QuotaConfig: {
                description: "Configuration related to quotas.",
                properties: {
                    signUpQuotaConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2TemporaryQuota",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2RequestLogging: {
                description: "Configuration for logging requests made to this project to Stackdriver Logging",
                properties: {
                    enabled: {
                        description: "Whether logging is enabled for this project or not.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2SendEmail: {
                description: "Options for email sending.",
                properties: {
                    callbackUri: { description: "action url in email template.", type: "string" },
                    changeEmailTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2EmailTemplate",
                    },
                    dnsInfo: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DnsInfo" },
                    legacyResetPasswordTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2EmailTemplate",
                    },
                    method: {
                        description: "The method used for sending an email.",
                        enum: ["METHOD_UNSPECIFIED", "DEFAULT", "CUSTOM_SMTP"],
                        type: "string",
                    },
                    resetPasswordTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2EmailTemplate",
                    },
                    revertSecondFactorAdditionTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2EmailTemplate",
                    },
                    smtp: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Smtp" },
                    verifyEmailTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2EmailTemplate",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2SendSms: {
                description: "Options for SMS sending.",
                properties: {
                    smsTemplate: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SmsTemplate",
                    },
                    useDeviceLocale: {
                        description: "Whether to use the accept_language header for SMS.",
                        type: "boolean",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2SignInConfig: {
                description: "Configuration related to local sign in methods.",
                properties: {
                    allowDuplicateEmails: {
                        description: "Whether to allow more than one account to have the same email.",
                        type: "boolean",
                    },
                    anonymous: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Anonymous" },
                    email: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Email" },
                    hashConfig: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2HashConfig" },
                    phoneNumber: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2PhoneNumber",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2SmsRegionConfig: {
                description: "Configures the regions where users are allowed to send verification SMS for the project or tenant. This is based on the calling code of the destination phone number.",
                properties: {
                    allowByDefault: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2AllowByDefault",
                    },
                    allowlistOnly: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2AllowlistOnly",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2SmsTemplate: {
                description: "The template to use when sending an SMS.",
                properties: {
                    content: {
                        description: "Output only. The SMS's content. Can contain the following placeholders which will be replaced with the appropriate values: %APP_NAME% - For Android or iOS apps, the app's display name. For web apps, the domain hosting the application. %LOGIN_CODE% - The OOB code being sent in the SMS.",
                        readOnly: true,
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Smtp: {
                description: "Configuration for SMTP relay",
                properties: {
                    host: { description: "SMTP relay host", type: "string" },
                    password: { description: "SMTP relay password", type: "string" },
                    port: { description: "SMTP relay port", format: "int32", type: "integer" },
                    securityMode: {
                        description: "SMTP security mode.",
                        enum: ["SECURITY_MODE_UNSPECIFIED", "SSL", "START_TLS"],
                        type: "string",
                    },
                    senderEmail: { description: "Sender email for the SMTP relay", type: "string" },
                    username: { description: "SMTP relay username", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2SpCertificate: {
                description: "The SP's certificate data for IDP to verify the SAMLRequest generated by the SP.",
                properties: {
                    expiresAt: {
                        description: "Timestamp of the cert expiration instance.",
                        format: "google-datetime",
                        type: "string",
                    },
                    x509Certificate: { description: "Self-signed public certificate.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2SpConfig: {
                description: "The SAML SP (Service Provider) configuration when the project acts as the relying party to receive and accept an authentication assertion issued by a SAML identity provider.",
                properties: {
                    callbackUri: {
                        description: "Callback URI where responses from IDP are handled.",
                        type: "string",
                    },
                    spCertificates: {
                        description: "Output only. Public certificates generated by the server to verify the signature in SAMLRequest in the SP-initiated flow.",
                        items: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SpCertificate" },
                        readOnly: true,
                        type: "array",
                    },
                    spEntityId: { description: "Unique identifier for all SAML entities.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2TemporaryQuota: {
                description: "Temporary quota increase / decrease",
                properties: {
                    quota: {
                        description: "Corresponds to the 'refill_token_count' field in QuotaServer config",
                        format: "int64",
                        type: "string",
                    },
                    quotaDuration: {
                        description: "How long this quota will be active for",
                        format: "google-duration",
                        type: "string",
                    },
                    startTime: {
                        description: "When this quota will take affect",
                        format: "google-datetime",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Tenant: {
                description: "A Tenant contains configuration for the tenant in a multi-tenant project.",
                properties: {
                    allowPasswordSignup: {
                        description: "Whether to allow email/password user authentication.",
                        type: "boolean",
                    },
                    autodeleteAnonymousUsers: {
                        description: "Whether anonymous users will be auto-deleted after a period of 30 days.",
                        type: "boolean",
                    },
                    client: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2ClientPermissionConfig",
                    },
                    disableAuth: {
                        description: "Whether authentication is disabled for the tenant. If true, the users under the disabled tenant are not allowed to sign-in. Admins of the disabled tenant are not able to manage its users.",
                        type: "boolean",
                    },
                    displayName: { description: "Display name of the tenant.", type: "string" },
                    emailPrivacyConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2EmailPrivacyConfig",
                    },
                    enableAnonymousUser: {
                        description: "Whether to enable anonymous user authentication.",
                        type: "boolean",
                    },
                    enableEmailLinkSignin: {
                        description: "Whether to enable email link user authentication.",
                        type: "boolean",
                    },
                    hashConfig: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2HashConfig" },
                    inheritance: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Inheritance",
                    },
                    mfaConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2MultiFactorAuthConfig",
                    },
                    monitoring: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2MonitoringConfig",
                    },
                    name: {
                        description: 'Output only. Resource name of a tenant. For example: "projects/{project-id}/tenants/{tenant-id}"',
                        readOnly: true,
                        type: "string",
                    },
                    smsRegionConfig: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2SmsRegionConfig",
                    },
                    testPhoneNumbers: {
                        additionalProperties: { type: "string" },
                        description: "A map of pairs that can be used for MFA. The phone number should be in E.164 format (https://www.itu.int/rec/T-REC-E.164/) and a maximum of 10 pairs can be added (error will be thrown once exceeded).",
                        type: "object",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitAdminV2Trigger: {
                description: "Synchronous Cloud Function with HTTP Trigger",
                properties: {
                    functionUri: { description: "HTTP URI trigger for the Cloud Function.", type: "string" },
                    updateTime: {
                        description: "When the trigger was changed.",
                        format: "google-datetime",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2AutoRetrievalInfo: {
                description: "The information required to auto-retrieve an SMS.",
                properties: {
                    appSignatureHash: {
                        description: "The Android app's signature hash for Google Play Service's SMS Retriever API.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2FinalizeMfaEnrollmentRequest: {
                description: "Finishes enrolling a second factor for the user.",
                properties: {
                    displayName: {
                        description: "Display name which is entered by users to distinguish between different second factors with same type or different type.",
                        type: "string",
                    },
                    idToken: { description: "Required. ID token.", type: "string" },
                    phoneVerificationInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaPhoneRequestInfo",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant that the user enrolling MFA belongs to. If not set, the user belongs to the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2FinalizeMfaEnrollmentResponse: {
                description: "FinalizeMfaEnrollment response.",
                properties: {
                    idToken: { description: "ID token updated to reflect MFA enrollment.", type: "string" },
                    phoneAuthInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaPhoneResponseInfo",
                    },
                    refreshToken: {
                        description: "Refresh token updated to reflect MFA enrollment.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2FinalizeMfaPhoneRequestInfo: {
                description: "Phone Verification info for a FinalizeMfa request.",
                properties: {
                    androidVerificationProof: {
                        description: 'Android only. Uses for "instant" phone number verification though GmsCore.',
                        type: "string",
                    },
                    code: { description: "User-entered verification code.", type: "string" },
                    phoneNumber: {
                        description: "Required if Android verification proof is presented.",
                        type: "string",
                    },
                    sessionInfo: {
                        description: "An opaque string that represents the enrollment session.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2FinalizeMfaPhoneResponseInfo: {
                description: "Phone Verification info for a FinalizeMfa response.",
                properties: {
                    androidVerificationProof: {
                        description: "Android only. Long-lived replacement for valid code tied to android device.",
                        type: "string",
                    },
                    androidVerificationProofExpireTime: {
                        description: "Android only. Expiration time of verification proof in seconds.",
                        format: "google-datetime",
                        type: "string",
                    },
                    phoneNumber: { description: "For Android verification proof.", type: "string" },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2FinalizeMfaSignInRequest: {
                description: "Finalizes sign-in by verifying MFA challenge.",
                properties: {
                    mfaPendingCredential: {
                        description: "Required. Pending credential from first factor sign-in.",
                        type: "string",
                    },
                    phoneVerificationInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaPhoneRequestInfo",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to. If not set, the user will sign in to the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2FinalizeMfaSignInResponse: {
                description: "FinalizeMfaSignIn response.",
                properties: {
                    idToken: { description: "ID token for the authenticated user.", type: "string" },
                    phoneAuthInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2FinalizeMfaPhoneResponseInfo",
                    },
                    refreshToken: {
                        description: "Refresh token for the authenticated user.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2StartMfaEnrollmentRequest: {
                description: "Sends MFA enrollment verification SMS for a user.",
                properties: {
                    idToken: { description: "Required. User's ID token.", type: "string" },
                    phoneEnrollmentInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaPhoneRequestInfo",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant that the user enrolling MFA belongs to. If not set, the user belongs to the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2StartMfaEnrollmentResponse: {
                description: "StartMfaEnrollment response.",
                properties: {
                    phoneSessionInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaPhoneResponseInfo",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2StartMfaPhoneRequestInfo: {
                description: "App Verification info for a StartMfa request.",
                properties: {
                    autoRetrievalInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2AutoRetrievalInfo",
                    },
                    iosReceipt: {
                        description: "iOS only. Receipt of successful app token validation with APNS.",
                        type: "string",
                    },
                    iosSecret: {
                        description: "iOS only. Secret delivered to iOS app via APNS.",
                        type: "string",
                    },
                    phoneNumber: {
                        description: "Required for enrollment. Phone number to be enrolled as MFA.",
                        type: "string",
                    },
                    recaptchaToken: { description: "Web only. Recaptcha solution.", type: "string" },
                    safetyNetToken: {
                        description: "Android only. Used to assert application identity in place of a recaptcha token. A SafetyNet Token can be generated via the [SafetyNet Android Attestation API](https://developer.android.com/training/safetynet/attestation.html), with the Base64 encoding of the `phone_number` field as the nonce.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2StartMfaPhoneResponseInfo: {
                description: "Phone Verification info for a StartMfa response.",
                properties: {
                    sessionInfo: {
                        description: "An opaque string that represents the enrollment session.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2StartMfaSignInRequest: {
                description: "Starts multi-factor sign-in by sending the multi-factor auth challenge.",
                properties: {
                    mfaEnrollmentId: {
                        description: "Required. MFA enrollment id from the user's list of current MFA enrollments.",
                        type: "string",
                    },
                    mfaPendingCredential: {
                        description: "Required. Pending credential from first factor sign-in.",
                        type: "string",
                    },
                    phoneSignInInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaPhoneRequestInfo",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant the user is signing in to. If not set, the user will sign in to the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2StartMfaSignInResponse: {
                description: "StartMfaSignIn response.",
                properties: {
                    phoneResponseInfo: {
                        $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV2StartMfaPhoneResponseInfo",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2WithdrawMfaRequest: {
                description: "Withdraws MFA.",
                properties: {
                    idToken: { description: "Required. User's ID token.", type: "string" },
                    mfaEnrollmentId: {
                        description: "Required. MFA enrollment id from a current MFA enrollment.",
                        type: "string",
                    },
                    tenantId: {
                        description: "The ID of the Identity Platform tenant that the user unenrolling MFA belongs to. If not set, the user belongs to the default Identity Platform project.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleCloudIdentitytoolkitV2WithdrawMfaResponse: {
                description: "Withdraws MultiFactorAuth response.",
                properties: {
                    idToken: {
                        description: "ID token updated to reflect removal of the second factor.",
                        type: "string",
                    },
                    refreshToken: {
                        description: "Refresh token updated to reflect removal of the second factor.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleIamV1AuditConfig: {
                description: 'Specifies the audit configuration for a service. The configuration determines which permission types are logged, and what identities, if any, are exempted from logging. An AuditConfig must have one or more AuditLogConfigs. If there are AuditConfigs for both `allServices` and a specific service, the union of the two AuditConfigs is used for that service: the log_types specified in each AuditConfig are enabled, and the exempted_members in each AuditLogConfig are exempted. Example Policy with multiple AuditConfigs: { "audit_configs": [ { "service": "allServices", "audit_log_configs": [ { "log_type": "DATA_READ", "exempted_members": [ "user:jose@example.com" ] }, { "log_type": "DATA_WRITE" }, { "log_type": "ADMIN_READ" } ] }, { "service": "sampleservice.googleapis.com", "audit_log_configs": [ { "log_type": "DATA_READ" }, { "log_type": "DATA_WRITE", "exempted_members": [ "user:aliya@example.com" ] } ] } ] } For sampleservice, this policy enables DATA_READ, DATA_WRITE and ADMIN_READ logging. It also exempts `jose@example.com` from DATA_READ logging, and `aliya@example.com` from DATA_WRITE logging.',
                properties: {
                    auditLogConfigs: {
                        description: "The configuration for logging of each type of permission.",
                        items: { $ref: "#/components/schemas/GoogleIamV1AuditLogConfig" },
                        type: "array",
                    },
                    service: {
                        description: "Specifies a service that will be enabled for audit logging. For example, `storage.googleapis.com`, `cloudsql.googleapis.com`. `allServices` is a special value that covers all services.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleIamV1AuditLogConfig: {
                description: 'Provides the configuration for logging a type of permissions. Example: { "audit_log_configs": [ { "log_type": "DATA_READ", "exempted_members": [ "user:jose@example.com" ] }, { "log_type": "DATA_WRITE" } ] } This enables \'DATA_READ\' and \'DATA_WRITE\' logging, while exempting jose@example.com from DATA_READ logging.',
                properties: {
                    exemptedMembers: {
                        description: "Specifies the identities that do not cause logging for this type of permission. Follows the same format of Binding.members.",
                        items: { type: "string" },
                        type: "array",
                    },
                    logType: {
                        description: "The log type that this config enables.",
                        enum: ["LOG_TYPE_UNSPECIFIED", "ADMIN_READ", "DATA_WRITE", "DATA_READ"],
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleIamV1Binding: {
                description: "Associates `members`, or principals, with a `role`.",
                properties: {
                    condition: { $ref: "#/components/schemas/GoogleTypeExpr" },
                    members: {
                        description: "Specifies the principals requesting access for a Google Cloud resource. `members` can have the following values: * `allUsers`: A special identifier that represents anyone who is on the internet; with or without a Google account. * `allAuthenticatedUsers`: A special identifier that represents anyone who is authenticated with a Google account or a service account. Does not include identities that come from external identity providers (IdPs) through identity federation. * `user:{emailid}`: An email address that represents a specific Google account. For example, `alice@example.com` . * `serviceAccount:{emailid}`: An email address that represents a Google service account. For example, `my-other-app@appspot.gserviceaccount.com`. * `serviceAccount:{projectid}.svc.id.goog[{namespace}/{kubernetes-sa}]`: An identifier for a [Kubernetes service account](https://cloud.google.com/kubernetes-engine/docs/how-to/kubernetes-service-accounts). For example, `my-project.svc.id.goog[my-namespace/my-kubernetes-sa]`. * `group:{emailid}`: An email address that represents a Google group. For example, `admins@example.com`. * `domain:{domain}`: The G Suite domain (primary) that represents all the users of that domain. For example, `google.com` or `example.com`. * `deleted:user:{emailid}?uid={uniqueid}`: An email address (plus unique identifier) representing a user that has been recently deleted. For example, `alice@example.com?uid=123456789012345678901`. If the user is recovered, this value reverts to `user:{emailid}` and the recovered user retains the role in the binding. * `deleted:serviceAccount:{emailid}?uid={uniqueid}`: An email address (plus unique identifier) representing a service account that has been recently deleted. For example, `my-other-app@appspot.gserviceaccount.com?uid=123456789012345678901`. If the service account is undeleted, this value reverts to `serviceAccount:{emailid}` and the undeleted service account retains the role in the binding. * `deleted:group:{emailid}?uid={uniqueid}`: An email address (plus unique identifier) representing a Google group that has been recently deleted. For example, `admins@example.com?uid=123456789012345678901`. If the group is recovered, this value reverts to `group:{emailid}` and the recovered group retains the role in the binding.",
                        items: { type: "string" },
                        type: "array",
                    },
                    role: {
                        description: "Role that is assigned to the list of `members`, or principals. For example, `roles/viewer`, `roles/editor`, or `roles/owner`.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleIamV1GetIamPolicyRequest: {
                description: "Request message for `GetIamPolicy` method.",
                properties: { options: { $ref: "#/components/schemas/GoogleIamV1GetPolicyOptions" } },
                type: "object",
            },
            GoogleIamV1GetPolicyOptions: {
                description: "Encapsulates settings provided to GetIamPolicy.",
                properties: {
                    requestedPolicyVersion: {
                        description: "Optional. The maximum policy version that will be used to format the policy. Valid values are 0, 1, and 3. Requests specifying an invalid value will be rejected. Requests for policies with any conditional role bindings must specify version 3. Policies with no conditional role bindings may specify any valid value or leave the field unset. The policy in the response might use the policy version that you specified, or it might use a lower policy version. For example, if you specify version 3, but the policy has no conditional role bindings, the response uses version 1. To learn which resources support conditions in their IAM policies, see the [IAM documentation](https://cloud.google.com/iam/help/conditions/resource-policies).",
                        format: "int32",
                        type: "integer",
                    },
                },
                type: "object",
            },
            GoogleIamV1Policy: {
                description: 'An Identity and Access Management (IAM) policy, which specifies access controls for Google Cloud resources. A `Policy` is a collection of `bindings`. A `binding` binds one or more `members`, or principals, to a single `role`. Principals can be user accounts, service accounts, Google groups, and domains (such as G Suite). A `role` is a named list of permissions; each `role` can be an IAM predefined role or a user-created custom role. For some types of Google Cloud resources, a `binding` can also specify a `condition`, which is a logical expression that allows access to a resource only if the expression evaluates to `true`. A condition can add constraints based on attributes of the request, the resource, or both. To learn which resources support conditions in their IAM policies, see the [IAM documentation](https://cloud.google.com/iam/help/conditions/resource-policies). **JSON example:** { "bindings": [ { "role": "roles/resourcemanager.organizationAdmin", "members": [ "user:mike@example.com", "group:admins@example.com", "domain:google.com", "serviceAccount:my-project-id@appspot.gserviceaccount.com" ] }, { "role": "roles/resourcemanager.organizationViewer", "members": [ "user:eve@example.com" ], "condition": { "title": "expirable access", "description": "Does not grant access after Sep 2020", "expression": "request.time < timestamp(\'2020-10-01T00:00:00.000Z\')", } } ], "etag": "BwWWja0YfJA=", "version": 3 } **YAML example:** bindings: - members: - user:mike@example.com - group:admins@example.com - domain:google.com - serviceAccount:my-project-id@appspot.gserviceaccount.com role: roles/resourcemanager.organizationAdmin - members: - user:eve@example.com role: roles/resourcemanager.organizationViewer condition: title: expirable access description: Does not grant access after Sep 2020 expression: request.time < timestamp(\'2020-10-01T00:00:00.000Z\') etag: BwWWja0YfJA= version: 3 For a description of IAM and its features, see the [IAM documentation](https://cloud.google.com/iam/docs/).',
                properties: {
                    auditConfigs: {
                        description: "Specifies cloud audit logging configuration for this policy.",
                        items: { $ref: "#/components/schemas/GoogleIamV1AuditConfig" },
                        type: "array",
                    },
                    bindings: {
                        description: "Associates a list of `members`, or principals, with a `role`. Optionally, may specify a `condition` that determines how and when the `bindings` are applied. Each of the `bindings` must contain at least one principal. The `bindings` in a `Policy` can refer to up to 1,500 principals; up to 250 of these principals can be Google groups. Each occurrence of a principal counts towards these limits. For example, if the `bindings` grant 50 different roles to `user:alice@example.com`, and not to any other principal, then you can add another 1,450 principals to the `bindings` in the `Policy`.",
                        items: { $ref: "#/components/schemas/GoogleIamV1Binding" },
                        type: "array",
                    },
                    etag: {
                        description: "`etag` is used for optimistic concurrency control as a way to help prevent simultaneous updates of a policy from overwriting each other. It is strongly suggested that systems make use of the `etag` in the read-modify-write cycle to perform policy updates in order to avoid race conditions: An `etag` is returned in the response to `getIamPolicy`, and systems are expected to put that etag in the request to `setIamPolicy` to ensure that their change will be applied to the same version of the policy. **Important:** If you use IAM Conditions, you must include the `etag` field whenever you call `setIamPolicy`. If you omit this field, then IAM allows you to overwrite a version `3` policy with a version `1` policy, and all of the conditions in the version `3` policy are lost.",
                        format: "byte",
                        type: "string",
                    },
                    version: {
                        description: "Specifies the format of the policy. Valid values are `0`, `1`, and `3`. Requests that specify an invalid value are rejected. Any operation that affects conditional role bindings must specify version `3`. This requirement applies to the following operations: * Getting a policy that includes a conditional role binding * Adding a conditional role binding to a policy * Changing a conditional role binding in a policy * Removing any role binding, with or without a condition, from a policy that includes conditions **Important:** If you use IAM Conditions, you must include the `etag` field whenever you call `setIamPolicy`. If you omit this field, then IAM allows you to overwrite a version `3` policy with a version `1` policy, and all of the conditions in the version `3` policy are lost. If a policy does not include any conditions, operations on that policy may specify any valid version or leave the field unset. To learn which resources support conditions in their IAM policies, see the [IAM documentation](https://cloud.google.com/iam/help/conditions/resource-policies).",
                        format: "int32",
                        type: "integer",
                    },
                },
                type: "object",
            },
            GoogleIamV1SetIamPolicyRequest: {
                description: "Request message for `SetIamPolicy` method.",
                properties: {
                    policy: { $ref: "#/components/schemas/GoogleIamV1Policy" },
                    updateMask: {
                        description: 'OPTIONAL: A FieldMask specifying which fields of the policy to modify. Only the fields in the mask will be modified. If no mask is provided, the following default mask is used: `paths: "bindings, etag"`',
                        format: "google-fieldmask",
                        type: "string",
                    },
                },
                type: "object",
            },
            GoogleIamV1TestIamPermissionsRequest: {
                description: "Request message for `TestIamPermissions` method.",
                properties: {
                    permissions: {
                        description: "The set of permissions to check for the `resource`. Permissions with wildcards (such as `*` or `storage.*`) are not allowed. For more information see [IAM Overview](https://cloud.google.com/iam/docs/overview#permissions).",
                        items: { type: "string" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleIamV1TestIamPermissionsResponse: {
                description: "Response message for `TestIamPermissions` method.",
                properties: {
                    permissions: {
                        description: "A subset of `TestPermissionsRequest.permissions` that the caller is allowed.",
                        items: { type: "string" },
                        type: "array",
                    },
                },
                type: "object",
            },
            GoogleProtobufEmpty: {
                description: "A generic empty message that you can re-use to avoid defining duplicated empty messages in your APIs. A typical example is to use it as the request or the response type of an API method. For instance: service Foo { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }",
                properties: {},
                type: "object",
            },
            GoogleTypeExpr: {
                description: 'Represents a textual expression in the Common Expression Language (CEL) syntax. CEL is a C-like expression language. The syntax and semantics of CEL are documented at https://github.com/google/cel-spec. Example (Comparison): title: "Summary size limit" description: "Determines if a summary is less than 100 chars" expression: "document.summary.size() < 100" Example (Equality): title: "Requestor is owner" description: "Determines if requestor is the document owner" expression: "document.owner == request.auth.claims.email" Example (Logic): title: "Public documents" description: "Determine whether the document should be publicly visible" expression: "document.type != \'private\' && document.type != \'internal\'" Example (Data Manipulation): title: "Notification string" description: "Create a notification string with a timestamp." expression: "\'New message received at \' + string(document.create_time)" The exact variables and functions that may be referenced within an expression are determined by the service that evaluates it. See the service documentation for additional information.',
                properties: {
                    description: {
                        description: "Optional. Description of the expression. This is a longer text which describes the expression, e.g. when hovered over it in a UI.",
                        type: "string",
                    },
                    expression: {
                        description: "Textual representation of an expression in Common Expression Language syntax.",
                        type: "string",
                    },
                    location: {
                        description: "Optional. String indicating the location of the expression for error reporting, e.g. a file name and a position in the file.",
                        type: "string",
                    },
                    title: {
                        description: "Optional. Title for the expression, i.e. a short string describing its purpose. This can be used e.g. in UIs which allow to enter the expression.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GrantTokenRequest: {
                properties: {
                    code: {
                        description: "ID token to exchange for an access token and a refresh token. This field is called `code` to conform with the OAuth 2.0 specification. This field is deprecated and is ignored.",
                        type: "string",
                    },
                    grantType: {
                        description: "The grant_types that are supported: - `refresh_token` to exchange a Identity Platform refresh_token for Identity Platform id_token/access_token and possibly a new Identity Platform refresh_token.",
                        type: "string",
                    },
                    refreshToken: {
                        description: "Identity Platform refresh_token. This field is ignored if `grantType` isn't `refresh_token`.",
                        type: "string",
                    },
                },
                type: "object",
            },
            GrantTokenResponse: {
                properties: {
                    access_token: { description: "DEPRECATED The granted access token.", type: "string" },
                    expires_in: {
                        description: "Expiration time of `access_token` in seconds.",
                        format: "int64",
                        type: "string",
                    },
                    id_token: { description: "The granted ID token", type: "string" },
                    project_id: {
                        description: "The client's project number",
                        format: "int64",
                        type: "string",
                    },
                    refresh_token: {
                        description: "The granted refresh token; might be the same as `refreshToken` in the request.",
                        type: "string",
                    },
                    token_type: {
                        description: "The type of `access_token`. Included to conform with the OAuth 2.0 specification; always `Bearer`.",
                        type: "string",
                    },
                    user_id: { description: "The local user ID", type: "string" },
                },
                type: "object",
            },
            EmulatorV1ProjectsConfig: {
                type: "object",
                description: "Emulator-specific configuration.",
                properties: {
                    signIn: { properties: { allowDuplicateEmails: { type: "boolean" } }, type: "object" },
                },
            },
            EmulatorV1ProjectsOobCodes: {
                type: "object",
                description: "Details of all pending confirmation codes.",
                properties: {
                    oobCodes: {
                        type: "array",
                        items: {
                            type: "object",
                            properties: {
                                email: { type: "string" },
                                oobCode: { type: "string" },
                                oobLink: { type: "string" },
                                requestType: { type: "string" },
                            },
                        },
                    },
                },
            },
            EmulatorV1ProjectsVerificationCodes: {
                type: "object",
                description: "Details of all pending verification codes.",
                properties: {
                    verificationCodes: {
                        type: "array",
                        items: {
                            type: "object",
                            properties: {
                                code: { type: "string" },
                                phoneNumber: { type: "string" },
                                sessionInfo: { type: "string" },
                            },
                        },
                    },
                },
            },
        },
        parameters: {
            access_token: {
                name: "access_token",
                in: "query",
                description: "OAuth access token.",
                schema: { type: "string" },
            },
            alt: {
                name: "alt",
                in: "query",
                description: "Data format for response.",
                schema: { type: "string", enum: ["json", "media", "proto"], default: "json" },
            },
            callback: { name: "callback", in: "query", description: "JSONP", schema: { type: "string" } },
            fields: {
                name: "fields",
                in: "query",
                description: "Selector specifying which fields to include in a partial response.",
                schema: { type: "string" },
            },
            oauth_token: {
                name: "oauth_token",
                in: "query",
                description: "OAuth 2.0 token for the current user.",
                schema: { type: "string" },
            },
            prettyPrint: {
                name: "prettyPrint",
                in: "query",
                description: "Returns response with indentations and line breaks.",
                schema: { type: "boolean", default: true },
            },
            quotaUser: {
                name: "quotaUser",
                in: "query",
                description: "Available to use for quota purposes for server-side applications. Can be any arbitrary string assigned to a user, but should not exceed 40 characters.",
                schema: { type: "string" },
            },
            uploadType: {
                name: "uploadType",
                in: "query",
                description: 'Legacy upload protocol for media (e.g. "media", "multipart").',
                schema: { type: "string" },
            },
            upload_protocol: {
                name: "upload_protocol",
                in: "query",
                description: 'Upload protocol for media (e.g. "raw", "multipart").',
                schema: { type: "string" },
            },
        },
        requestBodies: {
            GoogleCloudIdentitytoolkitV1BatchDeleteAccountsRequest: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1BatchDeleteAccountsRequest",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1SignUpRequest: {
                content: {
                    "application/json": {
                        schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SignUpRequest" },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1DeleteAccountRequest: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1DeleteAccountRequest",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1UploadAccountRequest: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1UploadAccountRequest",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1CreateSessionCookieRequest: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1CreateSessionCookieRequest",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1GetAccountInfoRequest: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetAccountInfoRequest",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1GetOobCodeRequest: {
                content: {
                    "application/json": {
                        schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1GetOobCodeRequest" },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1SetAccountInfoRequest: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1SetAccountInfoRequest",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitV1QueryUserInfoRequest: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitV1QueryUserInfoRequest",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitAdminV2Tenant: {
                content: {
                    "application/json": {
                        schema: { $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2Tenant" },
                    },
                },
            },
            GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2InboundSamlConfig",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2DefaultSupportedIdpConfig",
                        },
                    },
                },
            },
            GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig: {
                content: {
                    "application/json": {
                        schema: {
                            $ref: "#/components/schemas/GoogleCloudIdentitytoolkitAdminV2OAuthIdpConfig",
                        },
                    },
                },
            },
        },
        securitySchemes: {
            Oauth2: {
                type: "oauth2",
                description: "Oauth 2.0 implicit authentication",
                flows: {
                    implicit: {
                        authorizationUrl: "https://accounts.google.com/o/oauth2/auth",
                        scopes: {
                            "https://www.googleapis.com/auth/cloud-platform": "See, edit, configure, and delete your Google Cloud data and see the email address for your Google Account.",
                            "https://www.googleapis.com/auth/firebase": "View and administer all your Firebase data and settings",
                        },
                    },
                    authorizationCode: {
                        authorizationUrl: "https://accounts.google.com/o/oauth2/auth",
                        tokenUrl: "https://accounts.google.com/o/oauth2/token",
                        scopes: {
                            "https://www.googleapis.com/auth/cloud-platform": "See, edit, configure, and delete your Google Cloud data and see the email address for your Google Account.",
                            "https://www.googleapis.com/auth/firebase": "View and administer all your Firebase data and settings",
                        },
                    },
                },
            },
            apiKeyQuery: {
                type: "apiKey",
                name: "key",
                in: "query",
                description: "API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.",
            },
            apiKeyHeader: {
                type: "apiKey",
                name: "x-goog-api-key",
                in: "header",
                description: "API key. Your API key identifies your project and provides you with API access, quota, and reports. Required unless you provide an OAuth 2.0 token.",
            },
        },
    },
};
