"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.equal = exports.toExtensionVersionName = exports.toExtensionName = exports.toExtensionVersionRef = exports.toExtensionRef = exports.parse = void 0;
const semver = require("semver");
const error_1 = require("../error");
const refRegex = new RegExp(/^([^/@\n]+)\/{1}([^/@\n]+)(@{1}([^\n]+)|)$/);
function parse(refOrName) {
    const ret = parseRef(refOrName) || parseName(refOrName);
    if (!ret || !ret.publisherId || !ret.extensionId) {
        throw new error_1.FirebaseError(`Unable to parse ${refOrName} as an extension ref`);
    }
    if (ret.version &&
        !semver.valid(ret.version) &&
        !semver.validRange(ret.version) &&
        !["latest", "latest-approved"].includes(ret.version)) {
        throw new error_1.FirebaseError(`Extension reference ${ret} contains an invalid version ${ret.version}.`);
    }
    return ret;
}
exports.parse = parse;
function parseRef(ref) {
    const parts = refRegex.exec(ref);
    if (parts && (parts.length === 5 || parts.length === 7)) {
        const publisherId = parts[1];
        const extensionId = parts[2];
        const version = parts[4];
        return { publisherId, extensionId, version };
    }
}
function parseName(name) {
    const parts = name.split("/");
    return {
        publisherId: parts[1],
        extensionId: parts[3],
        version: parts[5],
    };
}
function toExtensionRef(ref) {
    return `${ref.publisherId}/${ref.extensionId}`;
}
exports.toExtensionRef = toExtensionRef;
function toExtensionVersionRef(ref) {
    if (!ref.version) {
        throw new error_1.FirebaseError(`Ref does not have a version`);
    }
    return `${ref.publisherId}/${ref.extensionId}@${ref.version}`;
}
exports.toExtensionVersionRef = toExtensionVersionRef;
function toExtensionName(ref) {
    return `publishers/${ref.publisherId}/extensions/${ref.extensionId}`;
}
exports.toExtensionName = toExtensionName;
function toExtensionVersionName(ref) {
    if (!ref.version) {
        throw new error_1.FirebaseError(`Ref does not have a version`);
    }
    return `publishers/${ref.publisherId}/extensions/${ref.extensionId}/versions/${ref.version}`;
}
exports.toExtensionVersionName = toExtensionVersionName;
function equal(a, b) {
    return (!!a &&
        !!b &&
        a.publisherId === b.publisherId &&
        a.extensionId === b.extensionId &&
        a.version === b.version);
}
exports.equal = equal;
